/* /lib/js/admin.js */
/**
 * Global form data storage for all forms in the plugin
 * @type {Object}
 */
let wc_form_data = [];
/**
 * WordPress and plugin configuration variables
 * @type {string}
 */
let wc_site_url, wc_api_loc, wc_nonce, wc_template_loc, wc_user_data, wc_page_class;
/**
 * Page UI elements
 * @type {HTMLElement}
 */
let page_title, page_container;
/**
 * Form reactivation delay in milliseconds
 * @constant {number}
 */
const reactivate_form_time = 500;
/**
 * String pattern used for API attribute path replacement
 * @constant {string}
 */
const replace_string = '/data/attributes/';
/**
 * Browser's locale language setting
 * @constant {string}
 */
const localLanguage = navigator.language ?? 'en-US';
/**
 * Current modal name identifier
 * @type {string}
 */
let modal_name;
/**
 * Checkup details tracking variables
 * @type {number|string}
 */
let checkup_details_id = 0, checkup_time_id, AuthTokenWebsiteCheckups;
/**
 * Loader timeout reference
 * @type {number|null}
 */
let loader_timeout_id = null;
/**
 * Popup button references for notifications and checkups
 * @type {HTMLElement}
 */
let btn_popup_manage_notification;
let btn_popup_manage_checkup;
/**
 * Counter for generating unique IDs
 * @type {number}
 */
let uid_counter = 0;

/**
 * Initializes and handles the user registration page
 * Sets up form fields, dropdowns, validation, and registration submission
 * @async
 * @returns {Promise<void>}
 */
async function registration_page() {
    try {
        const btn_register = getElById('btn_register');
        const btn_advanced = getElById('btn_advanced');
        const dropdown_language_data = {};
        const dropdown_country_data = {};

        const [language_result_popular, language_result_all] = await Promise.all([manage_api_data('languages?filter[popular]=1', 'get'), manage_api_data('languages', 'get')]);
        const [country_result_popular, country_result_all] = await Promise.all([manage_api_data('countries?filter[popular]=1', 'get'), manage_api_data('countries', 'get')]);
        const product_list = await manage_api_data('subproducts', 'get');

        const langDefault = wc_user_data['site_lang'] ?? navigator.language.substring(0, 2);
        const countryDefault = getDefaultCountry(country_result_all.data);

        clear_all_field(wc_form_data['simpleregisters']);

        dropdown_language_data.popular = generate_dropdown_list(language_result_popular.data, 'Please choose', 'language', langDefault);
        dropdown_language_data.all = generate_dropdown_list(language_result_all.data, 'Please choose', 'language', langDefault);

        dropdown_country_data.popular = generate_country_dropdown(country_result_popular.data, 'Please choose', countryDefault, false);
        dropdown_country_data.all = generate_country_dropdown(country_result_all.data, 'Please choose', countryDefault, true);

        let dropdown_language, switch_language, dropdown_country, switch_country, dropdown_subscription,
            dropdown_max, checkbox_auto_renew, checkbox_accept_term;

        wc_form_data['simpleregisters'].forEach((form_data, index) => {
            const field_name = form_data.field_name;
            const field_item = form_data.field_item;
            if (form_data?.field_advanced == 1) {
                const el = getParent(field_item, 3);
                setClass(el, '', 'hidden_toggle');
                setClass(el, '', 'd-none');
            }

            if (field_name === 'companyNameReg') {
                setTimeout(function () {
                    form_data.field_item.focus();
                }, 500);
            }

            if (field_name === 'emailReg') {
                if (wc_user_data['mail'] !== '') {
                    form_data.field_item.value = wc_user_data['mail'];
                }
            }

            if (field_name === 'firstnameReg') {
                if (wc_user_data['first_name'] !== '') {
                    form_data.field_item.value = wc_user_data['first_name'];
                }
            }
            if (field_name === 'lastnameReg') {
                if (wc_user_data['last_name'] !== '') {
                    form_data.field_item.value = wc_user_data['last_name'];
                }
            }
            if (field_name === 'websiteReg') {
                if (wc_user_data['site_address'] !== '') {
                    form_data.field_item.value = wc_user_data['site_address'];
                }
            }

            if (field_name === 'langReg') {
                dropdown_language = field_item;
            }
            if (field_name === 'switch_langReg') {
                switch_language = field_item;
            }
            if (field_name === 'countryReg') {
                dropdown_country = field_item;
            }
            if (field_name === 'switch_countryReg') {
                switch_country = field_item;
            }
            if (field_name === 'subproductIdReg') {
                dropdown_subscription = field_item;
            }
            if (field_name === 'maxReg') {
                dropdown_max = field_item;
            }
            if (field_name === 'periodReg') {
                dropdown_period = field_item;
            }
            if (field_name === 'autoRenewReg') {
                checkbox_auto_renew = field_item;
            }
            if (field_name === 'acceptedTermsConditionsReg') {
                checkbox_accept_term = field_item;

                wc_form_data['simpleregisters'][index]['field_required'] = false;
                wc_form_data['simpleregisters'][index]['detail'] = undefined;
            }
        });

        if (is_contained(langDefault, language_result_popular.data)) {
            dropdown_language.innerHTML = dropdown_language_data.popular;
            switch_language.checked = false;
        } else {
            dropdown_language.innerHTML = dropdown_language_data.all;
            switch_language.checked = true;
            switch_language.value = 1;
        }
        toggle_switch(switch_language, dropdown_language, dropdown_language_data.popular, dropdown_language_data.all);

        if (is_contained(countryDefault, country_result_popular.data)) {
            dropdown_country.innerHTML = dropdown_country_data.popular;
            switch_country.checked = false;
        } else {
            dropdown_country.innerHTML = dropdown_country_data.all;
            switch_country.checked = true;
            switch_country.value = 1;
        }
        toggle_switch(switch_country, dropdown_country, dropdown_country_data.popular, dropdown_country_data.all);

        dropdown_subscription.innerHTML = generate_subscription_lst(product_list.data);
        dropdown_max.innerHTML = generate_subscription_options(product_list.data, dropdown_subscription?.value ?? 1);
        dropdown_period.innerHTML = get_period_list(
            get_subscription_attr(product_list, dropdown_subscription?.value ?? 1)?.priceOption,
            '1y'
        );
        subscription_info(product_list, dropdown_subscription.value, dropdown_period?.value ?? '1y');

        dropdown_subscription.addEventListener('change', async function (e) {
            e.preventDefault();
            subscription_info(product_list, this.value, dropdown_period?.value ?? '1y');
            dropdown_max.innerHTML = generate_subscription_options(product_list.data, this.value ?? 1);
            dropdown_period.innerHTML = get_period_list(
                get_subscription_attr(product_list, this.value ?? 1)?.priceOption,
                dropdown_period.value ?? '1y'
            );

        });
        dropdown_period.addEventListener('change', async function (e) {
            e.preventDefault();
            subscription_info(product_list, dropdown_subscription.value, this.value ?? '1y');
        });

        checkbox_auto_renew.value = 1;
        checkbox_accept_term.value = 0;

        checkbox_auto_renew.checked = true;
        checkbox_auto_renew.addEventListener('change', function () {
            this.value = this.checked ? 1 : 0;
        });

        checkbox_accept_term.addEventListener('change', function () {
            this.value = this.checked ? 1 : 0;
        });
        btn_advanced.addEventListener('click', function (e) {
            e.preventDefault();
            const state = this.getAttribute('aria-pressed') == 'true';
            toggle_advanced_fields(state);
        });

        btn_register.addEventListener('click', function (e) {
            e.preventDefault();
            toggle_disable_frm(this);
            remove_all_error_class();

            if (!field_error_check(wc_form_data['simpleregisters'])) {
                const post_data = {};

                if (checkbox_accept_term.value === '1') {
                    wc_form_data['simpleregisters'].forEach(form_data => {
                        if (!['switch', 'textblock'].includes(form_data.field_type)) {
                            const field_name = form_data?.field_name;

                            if (field_name === 'isAdminPluginReg') {
                                post_data[field_name] = '1';
                            } else if (field_name === 'instanceReg') {
                                post_data[field_name] = 'wordpress';
                            } else if (field_name === 'timeZoneReg') {
                                post_data[field_name] = Intl.DateTimeFormat().resolvedOptions().timeZone;
                            } else if (field_name) {
                                if (field_name === 'autoRenewReg') {
                                    post_data[field_name] = parseInt(form_data.field_item.value.trim())
                                } else {
                                    post_data[field_name] = form_data.field_item.value.trim();
                                }
                            }
                        }
                    });

                    manage_api_data('simpleregisters', 'post', post_data)
                        .then(result => {
                            if (!result?.meta) {
                                generate_multiple_toast(result?.errors);
                                setTimeout(function () {
                                    toggle_disable_frm(btn_register);
                                    close_loader_safe();
                                }, reactivate_form_time);
                            } else {
                                post_data.token = result.data.attributes.token;
                                post_data.notice = result.meta.notice;
                                post_data.post_type = 'user_register';

                                manage_local_data(post_data);
                            }
                        })
                        .catch(error => {
                            console.error('Registration page error:', error);
                        });
                } else {
                    setClass(checkbox_accept_term, '', 'is-invalid');
                    show_toast(false, 'Please Accept Terms and conditions & Privacy Policy');
                    setTimeout(function () {
                        toggle_disable_frm(btn_register);
                    }, reactivate_form_time);
                }

            } else {
                setTimeout(function () {
                    toggle_disable_frm(btn_register);
                }, reactivate_form_time);
            }
        });
    } catch (error) {
        console.error('Registration submission error:', error);
    }
}

/**
 * Initializes and handles the settings page with tabbed interface
 * Manages tab switching, permissions, and saved tab state
 * @async
 * @returns {Promise<void>}
 */
async function setting_page() {
    let active_tab = get_webcheckups_storage_data('active_settings_tab') || '';

    const setting_tab_list = getAllElements('button[data-bs-toggle="tab"]');
    const setting_tab_list_arr = Array.from(setting_tab_list);

    for (const setting_tab of setting_tab_list_arr) {
        const index = setting_tab_list_arr.indexOf(setting_tab);
        if (index === 0) {
            const default_tab = setting_tab.getAttribute('data-bs-target').split('#')[1];
            // Only set if no saved tab is present
            if (!active_tab) {
                active_tab = default_tab;
            }
        }
        setting_tab.addEventListener('shown.bs.tab', async function (event) {
            active_tab = event.target.getAttribute('data-bs-target').split('#')[1];
            // Save tab state
            set_webcheckups_storage_data('active_settings_tab', active_tab);
            await change_tab_setting(active_tab);
        });

        if (wc_user_data['token'].length === 0) {
            if (index !== 0) {
                setting_tab.remove();
            }
        } else {
            const registration_exists = await check_data_exists('registrations');

            if (!registration_exists) {
                setting_tab_list_arr[2].remove();
                setting_tab_list_arr[3].remove();
            } else {
                const invoices_exists = await check_data_exists('invoices');
                if (!invoices_exists) {
                    setting_tab_list_arr[3].remove();
                }
            }
        }
        // Activate saved tab
        if (active_tab) {
            const tab_button = document.querySelector(`button[data-bs-target="#${active_tab}"]`);
            if (tab_button) {
                // Programmatically activate Bootstrap tab
                const tab = new bootstrap.Tab(tab_button);
                tab.show();
                await change_tab_setting(active_tab);
            } else {
                // Fallback: Load first tab if saved tab doesn't exist
                const first_tab = setting_tab_list_arr[0].getAttribute('data-bs-target').split('#')[1];
                active_tab = first_tab;
                await change_tab_setting(first_tab);            }
        }
    }
}

/**
 * Initializes and handles the notifications management page
 * Sets up notification form, modal, and submit handlers
 * @async
 * @returns {Promise<void>}
 */
async function notification_page() {
    clear_all_field(wc_form_data['notifications']);

    const btn_add_notifications_modal = getElById('btn_add_notifications_modal');
    btn_popup_manage_notification = getElById('btn_popup_manage_notification');

    const product_list = await manage_api_data('subproducts', 'get');
    const subscription_data = await manage_api_data('subscriptions', 'get');

    let dropdown_notification_type;

    await load_notification_data();

    wc_form_data['notifications'].forEach(form_data => {
        const field_name = form_data.field_name;

        if (field_name === 'notificationType') {
            dropdown_notification_type = form_data.field_item;
            dropdown_notification_type.innerHTML = '';

            const tmp_channel_data = product_list.data.find(item => parseInt(item.id) === subscription_data.data.attributes.subproductId).attributes.channel;
            generate_filter_dropdown(tmp_channel_data, dropdown_notification_type, form_data.values_description);
            add_tooltip_txt(dropdown_notification_type, dropdown_notification_type.options[dropdown_notification_type.selectedIndex].getAttribute('data-tooltip'));
        }

        form_data.field_item.addEventListener('change', async function (e) {
            e.preventDefault();
            if (form_data.field_type === 'select' && form_data?.values_description) {
                add_tooltip_txt(this, this.options[this.selectedIndex].getAttribute('data-tooltip'));
            }
        });
    });
    // Attach error clearing listeners AFTER fields are processed
    field_error_clear(wc_form_data['notifications']);

    btn_add_notifications_modal.addEventListener('click', async function (e) {
        e.preventDefault();
        btn_popup_manage_notification.setAttribute('data-method', 'post');
        btn_popup_manage_notification.removeAttribute('data-id');
        reset_form_field();
        clear_all_form();
        dropdown_notification_type.selectedIndex = 0;
    })

    btn_popup_manage_notification.addEventListener('click', async function (e) {
        e.preventDefault();
        toggle_disable_frm(this);
        remove_all_error_class();

        if (!field_error_check(wc_form_data['notifications'])) {
            const post_data = {};

            wc_form_data['notifications'].forEach(form_data => {
                const field_name = form_data.field_name;
                post_data[field_name] = form_data.field_item.value.trim();
            });

            if (this.getAttribute('data-method') === 'patch') {
                post_data.item_id = this.getAttribute('data-item_id');
            }
            const notification_post_data = await manage_api_data('notifications', this.getAttribute('data-method'), post_data);

            handle_toast(notification_post_data);
            if (notification_post_data?.errors) {
                toggle_disable_frm(btn_popup_manage_notification);
            } else {
                close_modal('modal_notifications');

                await load_notification_data().then(() => {
                    reset_form_field();
                    toggle_disable_frm(btn_popup_manage_notification);
                }).catch(error => {
                    console.error('Notification data load error:', error);
                })
            }
        } else {
            toggle_disable_frm(btn_popup_manage_notification);
        }
    })
}

/**
 * Initializes and handles the checkups management page
 * Sets up checkup form, dropdowns, validation, and submission
 * @async
 * @returns {Promise<void>}
 */
async function checkup_page() {
    clear_all_field(wc_form_data['sitecheckups']);
    const btn_add_sitecheckups_modal = getElById('btn_add_sitecheckups_modal');
    btn_popup_manage_checkup = getElById('btn_popup_manage_checkup');
    btn_advanced_fields = getElById('sitecheckups_block_advanced_btn');

    let field_checkup_name, dropdown_checkup_interval, dropdown_checkup_request_type,
        dropdown_checkup_notification_list, field_notify_After, field_wait, dropdown_checkup_authtype;

    const product_list = await manage_api_data('subproducts', 'get');
    const subscription_data = await manage_api_data('subscriptions', 'get');
    const notification_data = await manage_api_data('notifications', 'get');
    const subProductId = subscription_data?.data?.attributes?.subproductId ?? 1;

    await load_checkup_data();

    wc_form_data['sitecheckups'].forEach(form_data => {
        const field_name = form_data.field_name;
        const field_item = form_data.field_item;

        if (field_name === 'name') {
            field_checkup_name = field_item;
        }

        if (field_name === 'notificationId') {
            dropdown_checkup_notification_list = field_item;
            dropdown_checkup_notification_list.innerHTML = generate_grouped_dropdown_list(notification_data.data,
                ['firstnameNotification', 'lastnameNotification', 'notificationValue'], true);

            // Apply Vue-style features
            // Count TOTAL visible rows (optgroups + options)
            const totalVisibleRows = count_select_visible_rows(dropdown_checkup_notification_list);
            const adaptiveSize = calculate_adaptive_select_size(totalVisibleRows);
            dropdown_checkup_notification_list.size = adaptiveSize;

            // Add helper text with stats
            const totalNotifications = notification_data.data.length; // Für Info-Text
            add_notification_helper_text(dropdown_checkup_notification_list, totalNotifications);
        }

        if (field_name === 'checkInterval') {
            const minInterval = parseInt(subscription_data.data.attributes.frequency);
            const tmp_interval_json_data = product_list.data.
                find(item => parseInt(item.id) === subProductId).attributes.checkIntervals.
                filter(item => item.value >= minInterval);

            dropdown_checkup_interval = field_item;
            dropdown_checkup_interval.innerHTML = generate_dropdown_list(tmp_interval_json_data, null, 'text', null, false);
        }

        if (field_name === 'authtype') {
            dropdown_checkup_authtype = field_item;
            const auth_options = [
                {'text': 'No authentication required', 'value': null},
                {'text': 'Basic authentication', 'value': 'basic'},
                {'text': 'Digest authentication', 'value': 'digest'},
                {'text': 'NTLM authentication', 'value': 'ntlm'},
                {'text': 'Bearer authentication', 'value': 'bearer'},
            ];
            dropdown_checkup_authtype.innerHTML = generate_dropdown_list(auth_options,null, 'text', null, false);
        }

        if (field_name === 'requestAs') {
            const request_types = product_list.data.
                find(item => parseInt(item.id) === subProductId).attributes.checkRequests.map(item => ({
                    'type': 'request_type',
                    'id': item.id,
                    'attributes': {
                        'item_name': item.name,
                        'notificationType': item.type + ' request',
                    },
            }));

            dropdown_checkup_request_type = field_item;
            dropdown_checkup_request_type.innerHTML = generate_grouped_dropdown_list(request_types, ['item_name']);

            dropdown_checkup_request_type.addEventListener('change', function (e) {
                e.preventDefault();
                toggle_extended(this.value);
            });
        }

        if (['verifySsl', 'active'].includes(field_name)) {
            field_item.value = false;
            field_item.checked = false;
            field_item.addEventListener('change', function () {
                this.value = this.checked ? 1 : 0;
            });
        }

        if (field_name === 'notifyAfter') {
            field_notify_After = field_item;
        }

        if (field_name === 'wait') {
            field_wait = field_item;
        }
    });

    // Attach error clearing listeners AFTER fields are processed
    field_error_clear(wc_form_data['sitecheckups']);

    btn_advanced_fields.addEventListener('click', async function (e){
            e.preventDefault();
            toggle_extended(dropdown_checkup_request_type.value ?? '');
        }
    );

    btn_add_sitecheckups_modal.addEventListener('click', async function (e) {
        e.preventDefault();
        btn_popup_manage_checkup.setAttribute('data-method', 'post');
        btn_popup_manage_checkup.removeAttribute('data-id');
        reset_form_field();
        clear_all_form();
        dropdown_checkup_interval.selectedIndex = 0;
        dropdown_checkup_request_type.selectedIndex = 0;
        dropdown_checkup_authtype.selectedIndex = 0;
        dropdown_checkup_notification_list.value = '';
        field_notify_After.value = 2;
        field_wait.value = 3;
        setTimeout(function () {
            field_checkup_name.focus();
            if (getElById('noChecksFrom') || getElById('noChecksUntil')) {
                jQuery('#noChecksFrom, #noChecksUntil').mask('00:00');
            }
        }, 500);

        wc_form_data['sitecheckups'].forEach(form_data => {
            if (['verifySsl', 'active'].includes(form_data.field_name)) {
                form_data.field_item.value = 0;
                form_data.field_item.checked = false;
            }
        });
        // first load
        btn_advanced_fields.setAttribute('aria-pressed', 'false');
        setClass(btn_advanced_fields, 'active');
        toggle_extended('GET');
    });

    btn_popup_manage_checkup.addEventListener('click', async function (e) {
        e.preventDefault();
        toggle_disable_frm(this);
        remove_all_error_class();
        let tmp_checkup_from_data = [];
        setTimeout(function () {
            if (getElById('noChecksFrom') || getElById('noChecksUntil')) {
                jQuery('#noChecksFrom, #noChecksUntil').mask('00:00');
            }
        }, 500);

        // first load
        if (filled_extended_fields('GET')) {
            btn_advanced_fields.setAttribute('aria-pressed', 'true');
            setClass(btn_advanced_fields, '', 'active');
        } else {
            btn_advanced_fields.setAttribute('aria-pressed', 'false');
            setClass(btn_advanced_fields, 'active');
        }
        toggle_extended('GET');

        if (!field_error_check(tmp_checkup_from_data)) {
            const post_data = {};
            wc_form_data['sitecheckups'].forEach(form_data => {
                const name = form_data.field_name;
                const item = form_data.field_item;
                post_data[name] = item.value.trim();

                if (name === 'notificationId') {
                    post_data[name] = Array.from(item.options).filter(option => option.selected).map(option => option.value);
                }
                if (['checkInterval', 'wait', 'responseCode', 'verifySsl', 'notifyAfter', 'active'].includes(name)) {
                    post_data[name] = parseInt(item.value.trim())
                }

                if (['verifySsl', 'active'].includes(name)) {
                    if (item.value.trim() === '') {
                        item.value = 0;
                        post_data[name] = item.value.trim();
                    }
                }

                if (['responseText', 'postData', 'headers'].includes(name)) {
                    post_data[name] = item.value.trim() === '' ? null : item.value.trim();
                }
            });

            if (this.getAttribute('data-method') === 'patch') {
                post_data.item_id = this.getAttribute('data-item_id');
            }

            toggle_disable_frm(btn_popup_manage_checkup);
            const site_checkups_post_data = await manage_api_data('sitecheckups', this.getAttribute('data-method'), post_data);

            handle_toast(site_checkups_post_data);
            if (!site_checkups_post_data?.errors) {
                close_modal('modal_sitecheckups');

                await load_checkup_data().then(() => {
                    reset_form_field();
                    toggle_disable_frm(btn_popup_manage_checkup);
                }).catch(error => {
                    console.error('Checkup data load error:', error);
                })
            }
        } else {
            toggle_disable_frm(btn_popup_manage_checkup);
        }
    });
}

/**
 * Initializes and handles the checkup details/results page
 * Displays detailed statistics and charts for individual checkups
 * Supports URL parameters for direct navigation from dashboard
 * @async
 * @returns {Promise<void>}
 */
async function checkup_details_page() {
    const dropdown_checkup_result_period = getElById('dropdown_checkup_result_period');
    const dropdown_checkup_list = getElById('dropdown_checkup_list');

    // Read URL parameters if present (from dashboard navigation)
    const urlParams = new URLSearchParams(window.location.search);
    const url_checkup_id = urlParams.get('checkup_id');
    const url_period = urlParams.get('period');

    // Load last selections from localStorage
    const saved_checkup_id = localStorage.getItem('webcheckups_last_checkup_id');
    const saved_period = localStorage.getItem('webcheckups_last_checkup_period');

    // Set global variables from URL if present
    if (url_checkup_id) {
        checkup_details_id = parseInt(url_checkup_id);
    }
    if (url_period) {
        checkup_time_id = url_period;
    }

    // If no URL parameters, use localStorage values
    if (!url_checkup_id && saved_checkup_id) {
        checkup_details_id = parseInt(saved_checkup_id);
    }
    if (!url_period && saved_period) {
        checkup_time_id = saved_period;
    }

    const checkup_data = await manage_api_data('sitecheckups', 'get');

    if (checkup_data?.data && checkup_data?.data.length > 0) {
        const active_checkups = checkup_data.data.filter(item => item.attributes.active === 1);

        dropdown_checkup_list.innerHTML = generate_dropdown_list(active_checkups, null, 'name');
        dropdown_checkup_list.selectedIndex = 0;

        // IMPORTANT: Update global variables from dropdown BEFORE using them
        // This ensures synchronization even after localStorage.clear()
        let dropdownCheckupId = dropdown_checkup_list.value;
        let dropdownPeriod = dropdown_checkup_result_period.value;

        if (checkup_details_id !== 0) {
            dropdown_checkup_list.value = checkup_details_id;
            dropdown_checkup_result_period.value = checkup_time_id;

            // Check if setting the value actually worked
            // (fails if checkup_details_id doesn't exist in dropdown)
            if (dropdown_checkup_list.value !== checkup_details_id.toString()) {
                console.warn(`Checkup ID ${checkup_details_id} not found in dropdown, using first option`);
                dropdown_checkup_list.selectedIndex = 0;
                dropdown_checkup_result_period.selectedIndex = 0;
            }
            
            // Clean URL after setting values (optional - removes parameters from URL bar)
            if (url_checkup_id || url_period) {
                const cleanUrl = window.location.pathname + '?page=checkup_results_menu';
                window.history.replaceState({}, '', cleanUrl);
            }
        } else {
            dropdown_checkup_list.selectedIndex = 0;
            dropdown_checkup_result_period.selectedIndex = 0;
        }
        // Validate that selected checkup_id actually exists in dropdown
        // If not (e.g. checkup was deleted), fall back to first option
        const actualCheckupId = dropdown_checkup_list.value;

        if (!actualCheckupId || actualCheckupId === '') {
            dropdown_checkup_list.selectedIndex = 0;
            dropdown_checkup_result_period.selectedIndex = 0;
        }

        // CRITICAL: Synchronize global variables with actual dropdown values
        // This is the single source of truth after all the logic above
        checkup_details_id = parseInt(dropdown_checkup_list.value);
        checkup_time_id = dropdown_checkup_result_period.value;

        // console.log(`✅ Synchronized: checkup_details_id=${checkup_details_id}, checkup_time_id=${checkup_time_id}`);
        
        // Save initial values to localStorage
        localStorage.setItem('webcheckups_last_checkup_id', dropdown_checkup_list.value);
        localStorage.setItem('webcheckups_last_checkup_period', dropdown_checkup_result_period.value);
        
        dropdown_checkup_result_period.addEventListener('change', async function (e) {
            e.preventDefault();
            // Save selection to localStorage
            localStorage.setItem('webcheckups_last_checkup_period', this.value);
            showGraphic(parseInt(dropdown_checkup_list.value.trim()), this.value.trim(), 'details_chart');
            await load_checkup_details(dropdown_checkup_list.value.trim(), this.value.trim());
        });

        dropdown_checkup_list.addEventListener('change', async function (e) {
            e.preventDefault();
            // Save selection to localStorage
            localStorage.setItem('webcheckups_last_checkup_id', this.value);
            showGraphic(parseInt(this.value.trim()), dropdown_checkup_result_period.value.trim(), 'details_chart');
            await load_checkup_details(parseInt(this.value.trim()), dropdown_checkup_result_period.value.trim());
        });

        showGraphic(parseInt(dropdown_checkup_list.value.trim()), dropdown_checkup_result_period.value.trim(), 'details_chart');
        await load_checkup_details(parseInt(dropdown_checkup_list.value.trim()), dropdown_checkup_result_period.value.trim());
        // Setup navigation event handlers
        setupChartNavigationHandlers();
    }
}

/**
* Initializes and handles the dashboard page
* Displays overview statistics, charts, and recent activity
* @async
* @returns {Promise<void>}
*/
async function dashboard_page() {
    const dropdown_dashboard_action = getElById('dropdown_dashboard_action');
    const btn_refresh_dashboard = getElById('btn-refresh-dashboard');

    dropdown_dashboard_action.addEventListener('change', async function (e) {
        e.preventDefault();
        await load_dashboard_list(this.value.trim(), this.options[this.selectedIndex].text);
    });

    // Refresh button handler
    btn_refresh_dashboard.addEventListener('click', async function (e) {
        e.preventDefault();

        // Disable button and show spinner
        this.disabled = true;
        this.classList.add('refreshing');

        await load_dashboard_list(dropdown_dashboard_action.value.trim(), dropdown_dashboard_action.options[dropdown_dashboard_action.selectedIndex].text);

        // Re-enable button and remove spinner
        this.disabled = false;
        this.classList.remove('refreshing');
    });
    await load_dashboard_list('day', 'last day');
}

/**
 * Handles settings tab content loading and form initialization
 * Loads different forms based on active tab (user details, registration, subscription, invoices)
 * @async
 * @param {string} active_tab - ID of the active tab
 * @returns {Promise<void>}
 */
async function change_tab_setting(active_tab) {
    clear_all_toast();
    clear_all_form();

    if (active_tab === 'tab_user_details') {
        const language_data = await manage_api_data('languages', 'get');
        const users_data = await manage_api_data('users', 'get');
        const users_fields = wc_form_data['users'] || [];

        if (users_fields.length === 0) {
            console.error('User form fields not initialized');
            close_loader_safe();
            return;
        }
        
        users_fields.forEach(form_data => {
            const field_name = form_data.field_name;
            const field_item = form_data.field_item;

            // Skip if element doesn't exist (e.g., virtual switches)
            if (!field_item) {
                console.warn('⚠️  Field element not found:', field_name);
                return;
            }
            if (field_name === 'lang') {
                field_item.innerHTML = generate_dropdown_list(language_data.data, null, 'language');
            }
            field_item.value = users_data.data.attributes[field_name];
        });
        getElById('tab_user_details').querySelector('fieldset').disabled = true;
    }

    if (active_tab === 'tab_registration') {
        let dropdown_country, switch_country, checkbox_accept_term;
        const btn_update_registration = getElById('btn_update_registration');

        const dropdown_country_data = {};
        const [country_result_popular, country_result_all] = await Promise.all([manage_api_data('countries?filter[popular]=1', 'get'), manage_api_data('countries', 'get')]);

        dropdown_country_data.popular = generate_country_dropdown(country_result_popular.data, 'Please choose', null, false);
        dropdown_country_data.all = generate_country_dropdown(country_result_all.data, 'Please choose', null, true);

        const registration_data = await manage_api_data('registrations', 'get');

        const registration_fields = wc_form_data['registrations'] || [];
        if (registration_fields.length === 0) {
            console.error('Registration form fields not initialized');
            close_loader_safe();
            return;
         }

        registration_fields.forEach((form_data, index) => {
            const field_name = form_data.field_name;
            const field_item = form_data.field_item;

            // Skip if element doesn't exist
            if (!field_item) {
                console.warn('⚠️  Field element not found:', field_name);
                return;
            }

            if (field_name === 'country') {
                dropdown_country = field_item;
                dropdown_country.innerHTML = dropdown_country_data.all;
            }

            if (field_name === 'switch_country') {
                switch_country = field_item;
            }

            if (field_name === 'acceptedTermsConditions') {
                checkbox_accept_term = field_item;

                wc_form_data['registrations'][index]['field_required'] = false;
                wc_form_data['registrations'][index]['detail'] = undefined;
            }

            field_item.value = registration_data.data.attributes[field_name];
        });

        switch_country.checked = true;
        toggle_switch(switch_country, dropdown_country, dropdown_country_data.popular, dropdown_country_data.all);

        checkbox_accept_term.addEventListener('change', function () {
            this.value = this.checked ? 1 : 0;
        });

        btn_update_registration.addEventListener('click', async function (e) {
            e.preventDefault();
            toggle_disable_frm(this);
            remove_all_error_class();

            if (!field_error_check(wc_form_data['registrations'])) {
                const post_data = {
                    'item_id': registration_data.data.id,
                };
                if (checkbox_accept_term.value === '1') {
                    wc_form_data['registrations'].forEach(form_data => {
                        if (!['switch', 'textblock'].includes(form_data.field_type)) {
                            const field_name = form_data.field_name;
                            post_data[field_name] = form_data.field_item.value.trim();
                        }
                    });
                    const registration_result = await manage_api_data('registrations', 'patch', post_data);

                    handle_toast(registration_result, 'User Details Updated Successfully');
                    if (!registration_result?.errors) {
                        if (!registration_result?.meta) {
                            checkbox_accept_term.checked = false;
                        } else {
                            setTimeout(function () {
                                location.reload();
                            }, 2000);
                        }
                        close_loader_safe();
                    }

                    toggle_disable_frm(btn_update_registration);
                } else {
                    setClass(checkbox_accept_term, '', 'is-invalid');
                    show_toast(false, 'Please Accept Terms and conditions & Privacy Policy');
                    setTimeout(function () {
                        toggle_disable_frm(btn_update_registration);
                    }, reactivate_form_time);
                }
            } else {
                toggle_disable_frm(btn_update_registration);
            }
        });
    }

    if (active_tab === 'tab_subscription') {
        const btn_update_subscription = getElById('btn_update_subscription');
        let dropdown_subscription, checkbox_auto_renew, dropdown_max;
        const product_list = await manage_api_data('subproducts', 'get');
        const subscription_data = await manage_api_data('subscriptions', 'get');
        const subscription_attr = subscription_data?.data?.attributes ?? [];

        const subscription_fields = wc_form_data['subscriptions'] || [];
        if (subscription_fields.length === 0) {
            console.error('Subscription form fields not initialized');
            close_loader_safe();
            return;
        }

        subscription_fields.forEach(form_data => {
            const field_name = form_data.field_name;
            const field_item = form_data.field_item;

            if (field_name === 'subproductId') {
                dropdown_subscription = field_item;
                dropdown_subscription.innerHTML = generate_subscription_lst(product_list.data, subscription_attr?.subproductId ?? 1);
                subscription_info(product_list, subscription_attr[field_name] ?? 1, subscription_attr?.period ?? '1y');
            }

            if (field_name === 'max') {
                dropdown_max = field_item;
                dropdown_max.innerHTML = generate_subscription_options(
                    product_list.data,
                    subscription_attr?.subproductId ?? 1,
                    subscription_attr?.max ?? 1
                );
            }

            if (field_name === 'period') {
                dropdown_period = field_item;
                dropdown_period.innerHTML = get_period_list(
                    get_subscription_attr(product_list, subscription_attr?.subproductId ?? 1)?.priceOption,
                    subscription_attr?.period ?? '1y'
                );
            }

            if (field_name === 'autoRenew') {
                checkbox_auto_renew = field_item;
            }

            field_item.value = subscription_attr[field_name];
            field_item.checked = subscription_attr[field_name];
        });

        dropdown_subscription.addEventListener('change', async function (e) {
            e.preventDefault();
            dropdown_period.innerHTML = get_period_list(
                get_subscription_attr(product_list, this.value ?? 1)?.priceOption,
                subscription_attr?.period ?? '1y'
            );
            dropdown_max.innerHTML = generate_subscription_options(
                product_list.data,
                this.value ?? 1,
                subscription_attr?.max);
            subscription_info(product_list, this.value ?? 1, dropdown_period?.value ?? '1y');
        });
        dropdown_period.addEventListener('change', async function (e) {
            e.preventDefault();
            subscription_info(product_list, dropdown_subscription.value, this.value ?? '1y');
        });

        btn_update_subscription.addEventListener('click', async function (e) {
            e.preventDefault();
            toggle_disable_frm(this);
            remove_all_error_class();

            const post_data = {
                'item_id': subscription_data.data.id,
            };

            wc_form_data['subscriptions'].forEach(form_data => {
                if (!['switch', 'textblock'].includes(form_data.field_type)) {
                    const field_name = form_data.field_name;
                    post_data[field_name] = form_data?.field_item?.value.trim();
                }
            });
            const subscription_result = await manage_api_data('subscriptions', 'patch', post_data);
            handle_toast(subscription_result);
            toggle_disable_frm(this);
        });
    }

    if (active_tab === 'tab_invoice') {
        const element_setting_plan_name = getElById('element_setting_plan_name');
        const element_setting_auto_renew = getElById('element_setting_auto_renew');
        const element_setting_invoice_date = getElById('element_setting_invoice_date');
        const tbl_setting_invoice_list = getElById('tbl_setting_invoice_list');
        const pay_invoice_text = getElById('block_settings_info_text');
        const invoice_data = await manage_api_data('invoices', 'get');

        if (invoice_data.data.length > 0) {
            const product_list = await manage_api_data('subproducts', 'get');

            const latest_invoice_data = invoice_data.data[Object.keys(invoice_data.data).length - 1];
            const subscription_data = await manage_api_data('subscriptions', 'get');

            element_setting_plan_name.innerText = get_subscription_attr(product_list, latest_invoice_data.attributes.subproductId)?.name;
            element_setting_invoice_date.innerHTML = format_date(latest_invoice_data.attributes.end);
            element_setting_auto_renew.innerText = subscription_data.data.attributes.autoRenew ? 'with' : 'without';

            await render_table(
                'tbl_setting_invoice_list',
                invoice_data.data,
                'tbl_column/template_tbl_column_settings_invoice.html',
                (template, invoice_tbl_data, index) => {
                    const attr = invoice_tbl_data.attributes;

                    // Payment link für offene Rechnungen
                    if (attr.status === 'open' && attr.paymentLink) {
                        pay_invoice_text.innerHTML = '<br />You can pay your open invoice #W' + invoice_tbl_data.id +
                            ' <a href="' + attr.paymentLink + '" target="_blank">here</a>.'
                    }

                    return template
                        .replace('{{bill_number}}', 'W' + invoice_tbl_data.id)
                        .replace('{{invoice_url}}', wc_site_url + attr.link)
                        .replace('{{invoice_date}}', format_date(attr.invoiceDate))
                        .replace('{{invoice_total}}', format_num(attr.total, 2))
                        .replace('{{subscription_plan}}', get_subscription_attr(product_list, attr.subproductId)?.name)
                        .replace('{{status}}', attr.status)
                        .replace('{{service_period}}', getTimespan(attr.start, attr.end));
                }
            );
        }
    }
    close_loader_safe();
}

/**
 * Loads and renders notification data table
 * Sets up edit, delete, resend, filter, and search functionality
 * @async
 * @returns {Promise<void>}
 */
async function load_notification_data() {
    const dropdown_notification_action = getElById('dropdown_notification_action');
    const btn_notification_action = getElById('btn_notification_action');
    const dropdown_notification_filter = getElById('dropdown_notification_filter');
    const btn_notification_filter = getElById('btn_notification_filter');
    const field_notification_search = getElById('field_notification_search');
    const btn_notification_search = getElById('btn_notification_search');
    const tbl_notification = getElById('tbl_notification');
    const notification_checkbox_select_all = getElById('notification_checkbox_select_all');

    notification_checkbox_select_all.checked = false;

    setTimeout(async function () {
        tbl_notification.querySelector('tbody').innerHTML = '';
        const notification_data = await manage_api_data('notifications', 'get');

        if (notification_data.data.length > 0) {
            filter_dropdown_items(notification_data.data, dropdown_notification_filter, 'notificationType');

            await render_table(
                'tbl_notification',
                notification_data.data,
                'tbl_column/template_tbl_column_notifications.html',
                (template, data, index) => template
                    .replace('{{item_id}}', data.id)
                    .replace('{{user_name}}', (data.attributes.firstnameNotification || '') + ' ' + (data.attributes.lastnameNotification || '<br/>'))
                    .replace('{{notification_type}}', data.attributes.notificationType)
                    .replace('{{notification_value}}', data.attributes.notificationValue)
                    .replace('{{resend_ok}}', data.attributes?.resendVerificationPossible ?
                        '' : ' class="d-none"')
                    .replace('{{verify_mark}}', data.attributes?.verifiedAt ? 'tick_mark' : 'close_mark')
            );

            const btn_notification_edit = getAllElements('.btn_notification_edit');
            const btn_notification_delete = getAllElements('.btn_notification_delete');
            const btn_notification_resend = getAllElements('.btn_notification_resend');
            const notification_check_box_single = getAllElements('.notification_check_box_single');

            toggle_all_check_box(notification_checkbox_select_all, notification_check_box_single);

            btn_notification_action.addEventListener('click', function (e) {
                e.preventDefault();
                let tmp_data = [];
                if (dropdown_notification_action.value.trim() === 'remove') {
                    notification_check_box_single.forEach(checkbox => {
                        if (checkbox.checked) {
                            tmp_data.push(parseInt(checkbox.closest('tr').getAttribute('data-item_id')));
                        }
                    });
                }

                if (tmp_data.length < 1) {
                    show_toast(false, 'No notifications selected', 1);
                    return;
                }

                const post_data = {
                    bulk_select: true,
                    bulk_type: dropdown_notification_action.value.trim(),
                    notificationId: tmp_data
                };

                manage_api_data('notifications', 'post', post_data)
                    .then(result => {
                        if (!result) {
                            show_toast(true, 'All Notifications Deleted Successfully', 1);
                            load_notification_data();
                        }
                    })
                    .catch(error => {
                        console.error('Notification action error:', error);
                    });
            });

            btn_notification_filter.addEventListener('click', function () {
                search_tbl(tbl_notification, dropdown_notification_filter.value.trim().toLowerCase(), [2]);
            });

            field_notification_search.addEventListener('input', function () {
                if (this.value === '') {
                    search_tbl(tbl_notification, '', [1, 3]);
                }
            });

            btn_notification_search.addEventListener('click', function () {
                search_tbl(tbl_notification, field_notification_search.value.trim().toLowerCase(), [1, 3]);
            });

            btn_notification_edit.forEach(edit_btn => {
                edit_btn.addEventListener('click', async function (e) {
                    e.preventDefault();
                    clear_all_field(wc_form_data['notifications']);
                    clear_all_form();
                    const tmp_item_id = this.closest('tr').getAttribute('data-item_id');
                    const details = await manage_api_data('notifications/' + tmp_item_id, 'get');

                    if (details['data'] !== null) {
                        reset_form_field();
                        btn_popup_manage_notification.setAttribute('data-method', 'patch');
                        btn_popup_manage_notification.setAttribute('data-item_id', tmp_item_id);

                        wc_form_data['notifications'].forEach(form_data => {
                            const field_name = form_data.field_name;
                            form_data.field_item.value = details.data.attributes[field_name];
                        });
                    }
                    close_loader_safe();
                });
            });

            btn_notification_delete.forEach(delete_btn => {
                delete_btn.addEventListener('click', async function (e) {
                    e.preventDefault();
                    if (confirm_window('Are you Sure Delete This Notification?')) {
                        const post_data = {
                            item_id: this.closest('tr').getAttribute('data-item_id')
                        };

                        const notification_delete_data = await manage_api_data('notifications', 'delete', post_data);
                        handle_toast(notification_delete_data, 'Notification Deleted Successfully');
                        await load_notification_data();
                    }
                });
            });
            btn_notification_resend.forEach(resend_btn => {
                resend_btn.addEventListener('click', async function (e) {
                    e.preventDefault();
                    if (confirm_window('Resend Verification Request?')) {
                        const item_id = this.closest('tr').getAttribute('data-item_id');
                        const notification_resend_data = await manage_api_data('notifications/resend/' + item_id, 'get');
                        handle_toast(notification_resend_data);
                        await load_notification_data();
                    }
                });
            });
        }
        notification_checkbox_select_all.checked = false;
        dropdown_notification_action.value = '';
        await check_main_menu();
        close_loader_safe();
    }, 500);

}

/**
 * Loads and renders checkup data table
 * Sets up edit, delete, filter, search, and bulk action functionality
 * @async
 * @returns {Promise<void>}
 */
async function load_checkup_data() {
    const dropdown_checkup_action = getElById('dropdown_checkup_action');
    const btn_checkup_action = getElById('btn_checkup_action');
    const dropdown_checkup_filter = getElById('dropdown_checkup_filter');
    const btn_checkup_filter = getElById('btn_checkup_filter');
    const field_checkup_search = getElById('field_checkup_search');
    const btn_checkup_search = getElById('btn_checkup_search');
    const tbl_checkup = getElById('tbl_checkup');
    const checkup_checkbox_select_all = getElById('checkup_checkbox_select_all');

    checkup_checkbox_select_all.checked = false;

    setTimeout(async function () {
        tbl_checkup.querySelector('tbody').innerHTML = '';
        const checkup_data = await manage_api_data('sitecheckups', 'get');

        if (checkup_data?.data && checkup_data.data.length > 0) {
            filter_dropdown_items(checkup_data.data, dropdown_checkup_filter, 'requestAs');

            await render_table(
                'tbl_checkup',
                checkup_data.data,
                'tbl_column/template_tbl_column_checkups.html',
                (template, data, index) => template
                    .replace('{{item_id}}', data.id)
                    .replace('{{checkup_type}}', data.attributes.requestAs)
                    .replace('{{checkup_name}}', data.attributes.name)
                    .replace('{{checkup_url}}', data.attributes.link)
                    .replace('{{checkup_interval}}', format_time(data.attributes.checkInterval))
                    .replace('{{active_mark}}', data.attributes?.active !== 0 ? 'tick_mark' : 'close_mark')
            );

            const btn_checkup_edit = getAllElements('.btn_checkup_edit');
            const btn_checkup_delete = getAllElements('.btn_checkup_delete');
            const checkup_check_box_single = getAllElements('.checkup_check_box_single');
            const btn_advanced_fields = getElById('sitecheckups_block_advanced_btn');

            toggle_all_check_box(checkup_checkbox_select_all, checkup_check_box_single);

            btn_checkup_action.addEventListener('click', function (e) {
                e.preventDefault();
                let tmp_data = [];
                if (dropdown_checkup_action.value.trim() === 'remove') {
                    tmp_data = [];
                    checkup_check_box_single.forEach(checkbox => {
                        if (checkbox.checked) {
                            tmp_data.push(parseInt(checkbox.closest('tr').getAttribute('data-item_id')));
                        }
                    });
                }

                if (dropdown_checkup_action.value.trim() === 'deactivate') {
                    tmp_data = [];
                    checkup_check_box_single.forEach(checkbox => {
                        if (checkbox.checked) {
                            tmp_data.push(checkbox.closest('tr').getAttribute('data-item_id'));
                        }
                    });
                }

                if (tmp_data.length < 1) {
                    show_toast(false, 'No checkups selected', 1);
                    return;
                }

                const post_data = {
                    bulk_select: true,
                    bulk_type: dropdown_checkup_action.value.trim(),
                    checkupId: tmp_data
                };

                manage_api_data('sitecheckups', 'post', post_data)
                    .then(result => {
                        if (!result) {
                            show_toast(true, 'All checkups ' + dropdown_checkup_action.value.trim() + ' successfully', 1);
                            load_checkup_data();
                        }
                    })
                    .catch(error => {
                        console.error('Checkup action error:', error);
                    });
            });

            btn_checkup_filter.addEventListener('click', function () {
                search_tbl(tbl_checkup, dropdown_checkup_filter.value.trim().toLowerCase(), [1]);
            });

            field_checkup_search.addEventListener('input', function () {
                if (this.value === '') {
                    search_tbl(tbl_checkup, '', [2, 3]);
                }
            });

            btn_checkup_search.addEventListener('click', function () {
                search_tbl(tbl_checkup, field_checkup_search.value.trim().toLowerCase(), [2, 3]);
            });

            btn_checkup_edit.forEach(edit_btn => {
                edit_btn.addEventListener('click', async function (e) {
                    e.preventDefault();

                    const tmp_item_id = this.closest('tr').getAttribute('data-item_id');
                    const single_checkup_details = await manage_api_data('sitecheckups/' + tmp_item_id, 'get');

                    setTimeout(function () {
                        if (getElById('noChecksFrom') || getElById('noChecksUntil')) {
                            jQuery('#noChecksFrom, #noChecksUntil').mask('00:00');
                        }
                    }, 500);

                    if (single_checkup_details['data'] !== null) {
                        reset_form_field();
                        clear_all_form();
                        remove_all_error_class();
                        btn_popup_manage_checkup.setAttribute('data-method', 'patch');
                        btn_popup_manage_checkup.setAttribute('data-item_id', tmp_item_id);

                        const attr = single_checkup_details.data.attributes;
                        wc_form_data['sitecheckups'].forEach(form_data => {
                            const field_name = form_data.field_name;
                            form_data.field_item.value = attr[field_name];

                            if (field_name === 'notificationId') {
                                attr[field_name].forEach(value => form_data.field_item.querySelector(`option[value='${value}']`).selected = true);
                            }

                            if (['verifySsl', 'active'].includes(field_name)) {
                                form_data.field_item.checked = parseInt(attr[field_name]) === 1;
                            }
                        });
                        if (filled_extended_fields(attr?.requestAs)) {
                            btn_advanced_fields.setAttribute('aria-pressed', 'true');
                            setClass(btn_advanced_fields, '', 'active');
                        } else {
                            btn_advanced_fields.setAttribute('aria-pressed', 'false');
                            setClass(btn_advanced_fields, 'active');
                        }
                        toggle_extended(attr?.requestAs);
                    }
                    close_loader_safe();
                });
            });

            btn_checkup_delete.forEach(delete_btn => {
                delete_btn.addEventListener('click', async function (e) {
                    e.preventDefault();
                    if (confirm_window('Are you Sure Delete This checkup?')) {
                        const post_data = {
                            item_id: this.closest('tr').getAttribute('data-item_id')
                        };

                        const checkup_delete_data = await manage_api_data('sitecheckups', 'delete', post_data);
                        handle_toast(checkup_delete_data, 'Checkup Deleted Successfully');
                        await load_checkup_data();
                    }
                });
            });
        }
        checkup_checkbox_select_all.checkbox = false;
        dropdown_checkup_action.value = '';
        await check_main_menu();
        close_loader_safe();
    }, 500);
}

/**
 * Loads and displays detailed checkup results and statistics
 * Shows performance metrics, SSL info, failures, and response times
 * @async
 * @param {string|number} checkup_id - ID of the checkup to load
 * @param {string} checkup_period - Time period for the data (e.g., 'day', 'week', 'month')
 * @returns {Promise<void>}
 */
async function load_checkup_details(checkup_id, checkup_period) {
    const block_no_item = document.querySelector('.block_no_item');
    const block_checkup_details = getElById('block_checkup_details');
    const tbl_checkup_result_slowest_response = getElById('tbl_checkup_result_slowest_response');
    const tbl_checkup_result_last_failure = getElById('tbl_checkup_result_last_failure');
    const tbl_checkup_result_fastest_response = getElById('tbl_checkup_result_fastest_response');

    const block_slowest = getElById('block_slowest');
    const block_fastest = getElById('block_fastest');
    const block_last_failures = getElById('block_last_failures');
    const block_no_failures = getElById('block_details_no_failures');

    const checkup_details = await manage_api_data('checkupdetails/' + checkup_id + '/' + checkup_period, 'get');

    const block_checkup_result_url = getElById('block_checkup_result_url');
    const block_checkup_result_interval = getElById('block_checkup_result_interval');
    const block_checkup_result_last_check_checkup_time = getElById('block_checkup_result_last_check_checkup_time');
    const block_checkup_result_response_time = getElById('block_checkup_result_response_time');
    const block_checkup_result_warning = getElById('block_checkup_result_warning');
    const block_checkup_result_online_since = getElById('block_checkup_result_online_since');
    const block_checkup_result_mail_ports = getElById('block_checkup_result_mail_ports');
    const block_checkup_result_smtp_port = getElById('block_checkup_result_smtp_port');

    const block_checkup_result_ssl_ssl_issuer = getElById('block_checkup_result_ssl_ssl_issuer');
    const block_checkup_result_ssl_cipher_protocol = getElById('block_checkup_result_ssl_cipher_protocol');
    const block_checkup_result_ssl_valid_from_to = getElById('block_checkup_result_ssl_valid_from_to');

    const block_checkup_result_dns_error = getElById('block_checkup_result_dns_error');
    const block_checkup_result_ip_address = getElById('block_checkup_result_ip_address');
    const block_checkup_result_minmax_time = getElById('block_checkup_result_minmax_time');
    const block_checkup_result_domain_expires = getElById('block_checkup_result_domain_expires');
    const block_checkup_successful_failed_checkup = getElById('block_checkup_successful_failed_checkup');

    const block_checkup_result_total_downtime = getElById('block_checkup_result_total_downtime');
    const checkup_status_text = getElById('checkup_status_text');
    const block_checkup_error_alert = getElById('block_checkup_error_alert');
    const block_checkup_error_message = getElById('block_checkup_error_message');
    const block_offline_since_container = getElById('block_offline_since_container');
    const block_offline_since = getElById('block_offline_since');
    const attr = checkup_details.data.attributes;

    if (checkup_details?.data) {
        setClass(block_checkup_details, 'd-none');
        setClass(block_no_item, '', 'd-none');

        const details_link = attr?.link;
        const details_check_interval = attr?.checkInterval;
        const details_last_check_performed = attr?.lastCheckupPerformed;
        const lastCheckupDescription = attr?.lastCheckupDescription;
        const details_online_since = attr?.onlineSince;
        const details_offline_since = attr?.offlineSince;
        const details_last_checkup_time = attr?.lastCheckupTime;
        const details_avg_response_time = attr?.avgResponseTime;

        const details_mail_ports = attr?.openPorts;
        const details_smtp_port = attr?.smtpPort;

        const details_ssl_status = attr?.sslStatus;
        const details_ssl_cipher = attr?.sslCipher;
        const details_ssl_valid_from = attr?.sslValidFrom;
        const details_ssl_valid_till = attr?.sslValidTill;
        const details_dns_error = attr?.DNSError;

        const details_ip = attr?.ip;
        const details_fastest_checkup = attr?.fastestCheckups ?? undefined;
        const details_slowest_checkup = attr?.slowestCheckups ?? undefined;
        const details_last_errors = attr?.failedList;
        const details_warning = attr?.statusInfo?.short === 'Warning' ? attr?.statusInfo?.desc : undefined;

        const details_domain_expire = attr?.domainExpires;
        const details_successful_checkup = attr?.checkupsOk;
        const details_failed_checkup = attr?.checkupsFailed;
        const details_total_downtime = attr?.totalDowntime;

        const isOffline = attr?.statusInfo?.short?.toLowerCase() === 'failure';

        checkup_status_text.textContent = get_status_text(attr?.statusInfo);
        checkup_status_text.className = get_status_class(attr?.statusInfo);

        if (isOffline && lastCheckupDescription) {
            block_checkup_error_message.textContent = lastCheckupDescription;
            setClass(block_checkup_error_alert, 'd-none');
        } else {
            setClass(block_checkup_error_alert, '', 'd-none');
        }

        if (isOffline && details_offline_since) {
            block_offline_since.textContent = format_date(details_offline_since);
            setClass(block_offline_since_container, 'd-none');
        } else {
            setClass(block_offline_since_container, '', 'd-none');
        }

        block_checkup_result_url.innerText = details_link;
        block_checkup_result_interval.innerText = 'Every ' + format_time(details_check_interval);
        block_checkup_result_last_check_checkup_time.innerText = (details_last_check_performed ? format_date(details_last_check_performed) + ', ' : '') + lastCheckupDescription;

        if (!isOffline && details_last_checkup_time && details_last_checkup_time > 0) {
            const perfIndicator = calculate_performance_indicator(details_last_checkup_time, details_avg_response_time);
            let responseTimeText = details_last_checkup_time.toFixed(3) + ' sec';

            if (perfIndicator.show) {
                responseTimeText += ` <span class="${perfIndicator.class} performance-indicator" title="${perfIndicator.tooltipText}">${perfIndicator.icon}</span>`;
            }

            block_checkup_result_response_time.innerHTML = responseTimeText;
            setClass(block_checkup_result_response_time.closest('li'), 'd-none');
        } else {
            setClass(block_checkup_result_response_time.closest('li'), '', 'd-none');
        }

        block_checkup_result_warning.innerText = details_warning;

        if (!isOffline && details_online_since) {
            block_checkup_result_online_since.innerText = format_date(details_online_since);
            setClass(block_checkup_result_online_since.closest('li'), 'd-none');
        } else {
            setClass(block_checkup_result_online_since.closest('li'), '', 'd-none');
        }

        block_checkup_result_mail_ports.innerText = details_mail_ports;
        block_checkup_result_smtp_port.innerText = details_smtp_port;

        block_checkup_result_ssl_ssl_issuer.innerText = details_ssl_status;
        block_checkup_result_ssl_cipher_protocol.innerText = details_ssl_cipher ?? '';
        block_checkup_result_ssl_valid_from_to.innerText = details_ssl_valid_from ? getTimespan(details_ssl_valid_from, details_ssl_valid_till) : '';

        block_checkup_result_dns_error.innerText = details_dns_error;
        block_checkup_result_ip_address.innerText = details_ip;
        block_checkup_result_domain_expires.innerText = details_domain_expire ? format_date(details_domain_expire) : '';
        block_checkup_successful_failed_checkup.innerText = format_num(details_successful_checkup, 0) +
            ' successful, ' + format_num(details_failed_checkup, 0) + ' failed';

        if (details_total_downtime && details_total_downtime > 0) {
            block_checkup_result_total_downtime.innerText = format_downtime(details_total_downtime);
            setClass(block_checkup_result_total_downtime.closest('li'), 'd-none');
        } else {
            setClass(block_checkup_result_total_downtime.closest('li'), '', 'd-none');
        }

        let details_minmax_checkup = false;
        if (details_fastest_checkup !== undefined && details_slowest_checkup !== undefined &&
            details_fastest_checkup != details_slowest_checkup) {
            block_checkup_result_minmax_time.innerText = 'Slowest: ' +
                get_most_recent_time(details_slowest_checkup, 'slow').time + ' sec' +
                ', Fastest: ' + get_most_recent_time(details_fastest_checkup, 'fast').time + ' sec';
            details_minmax_checkup = true;
        }

        const shouldShowSSLValid = details_ssl_valid_from && details_ssl_valid_till;

        block_checkup_result_url.closest('li').classList.toggle('d-none', !details_link);
        block_checkup_result_interval.closest('li').classList.toggle('d-none', !details_check_interval);
        block_checkup_result_warning.closest('li').classList.toggle('d-none', !details_warning);
        block_checkup_result_mail_ports.closest('li').classList.toggle('d-none', !details_mail_ports);
        block_checkup_result_smtp_port.closest('li').classList.toggle('d-none', !details_smtp_port);
        block_checkup_result_ssl_ssl_issuer.closest('li').classList.toggle('d-none', !details_ssl_status);
        block_checkup_result_ssl_cipher_protocol.closest('li').classList.toggle('d-none', !details_ssl_cipher);
        block_checkup_result_ssl_valid_from_to.closest('li').classList.toggle('d-none', !shouldShowSSLValid);

        block_checkup_result_dns_error.closest('li').classList.toggle('d-none',
            details_dns_error === '' || details_dns_error === undefined || details_dns_error === false);

        block_checkup_result_ip_address.closest('li').classList.toggle('d-none',
            details_ip === '' || details_ip === undefined);
        block_checkup_result_domain_expires.closest('li').classList.toggle('d-none',
            details_domain_expire === '' || details_domain_expire === undefined);
        block_checkup_result_minmax_time.closest('li').classList.toggle('d-none', !details_minmax_checkup);

        if (details_slowest_checkup && details_last_check_performed) {
            setClass(block_slowest, 'd-none');
            await render_table(
                'tbl_checkup_result_slowest_response',
                details_slowest_checkup,
                'tbl_column/template_tbl_column_checkup_results.html',
                (template, data, index) => template
                    .replace('{{index_item}}', (index + 1))
                    .replace('{{checkup_results_date}}', format_date(data.performed, false))
                    .replace('{{checkup_results_result}}', data.time + ' sec')
            );
        } else {
            setClass(block_slowest, '', 'd-none');
            tbl_checkup_result_slowest_response.querySelector('tbody').innerHTML = '';
            setClass(block_slowest, '', 'd-none');
        }

        if (details_last_errors) {
            setClass(block_no_failures, '', 'd-none');
            setClass(tbl_checkup_result_last_failure, 'd-none');
            await render_table(
                'tbl_checkup_result_last_failure',
                details_last_errors,
                'tbl_column/template_tbl_column_checkup_failures.html',
                (template, data, index) => {
                    const dateRow = getTimespan(data.from, data.until, false);
                    const shortDesc = data.description.length <= 50 ? data.description :
                        data.description.substring(0, 50) + "\u2026";
                    const fullDesc = data.description.length <= 50 ? '' : data.description;
                    const times = format_num(data.checkups, 0) + ' time' + (data.checkups > 1 ? 's' : '');

                    return cleanEmptyToolTip(
                        template
                            .replace('{{index_item}}', (index + 1))
                            .replace('{{checkup_results_date}}', dateRow)
                            .replace('{{checkup_results_times}}', times)
                            .replace('{{checkup_results_short}}', shortDesc)
                            .replace('{{checkup_results_full}}', fullDesc)
                    );
                }
            );
        } else {
            setClass(block_no_failures, 'd-none');
            tbl_checkup_result_last_failure.querySelector('tbody').innerHTML = '';
            setClass(tbl_checkup_result_last_failure, '', 'd-none');
        }

        if (details_fastest_checkup && details_last_check_performed) {
            setClass(block_fastest, 'd-none');

            await render_table(
                'tbl_checkup_result_fastest_response',
                details_fastest_checkup,
                'tbl_column/template_tbl_column_checkup_results.html',
                (template, data, index) => template
                    .replace('{{index_item}}', (index + 1))
                    .replace('{{checkup_results_date}}', format_date(data.performed, false))
                    .replace('{{checkup_results_result}}', data.time + ' sec')
            );
        } else {
            tbl_checkup_result_fastest_response.querySelector('tbody').innerHTML = '';
            setClass(block_fastest, '', 'd-none');
        }
    } else {
        setClass(block_checkup_details, '', 'd-none');
        setClass(block_no_item, 'd-none');
        setClass(block_no_failures, 'd-none');
        setClass(tbl_checkup_result_last_failure, '', 'd-none');

        block_checkup_result_url.closest('li').classList.add('d-none');
        block_checkup_result_interval.closest('li').classList.add('d-none');
        block_checkup_result_last_check_checkup_time.closest('li').classList.add('d-none');
        block_checkup_result_warning.closest('li').classList.add('d-none');
        block_checkup_result_online_since.closest('li').classList.add('d-none');
        block_checkup_result_mail_ports.closest('li').classList.add('d-none');
        block_checkup_result_smtp_port.closest('li').classList.add('d-none');

        block_checkup_result_ssl_ssl_issuer.closest('li').classList.add('d-none');
        block_checkup_result_ssl_cipher_protocol.closest('li').classList.add('d-none');
        block_checkup_result_ssl_valid_from_to.closest('li').classList.add('d-none');

        block_checkup_result_dns_error.closest('li').classList.add('d-none');
        block_checkup_result_ip_address.closest('li').classList.add('d-none');
        block_checkup_result_domain_expires.closest('li').classList.add('d-none');
        block_checkup_result_minmax_time.closest('li').classList.add('d-none');
        block_checkup_successful_failed_checkup.closest('li').classList.add('d-none');
    }
    close_loader_safe();
}

// ===================================
// SLIDER LAZY LOADING WITH INTERSECTION OBSERVER
// ===================================

/**
  * Initializes lazy loading for dashboard slider
  *
  * @description
  * Uses Intersection Observer to pause/resume slider when outside viewport.
  * Prevents unnecessary CPU/GPU usage when slider is not visible.
  *
  * **Performance benefits:**
  * - Pauses auto-rotation when slider scrolled out of view
  * - Resumes when slider becomes visible again
  * - Minimal overhead (~1-2ms)
  */
function initSliderLazyLoading() {
    const sliderElement = document.querySelector('#graph_slider');
    if (!sliderElement) return;

    // console.log('🎬 Initializing slider lazy loading');

    // Check if Intersection Observer is supported
    if (!('IntersectionObserver' in window)) {
        console.warn('⚠️ IntersectionObserver not supported, slider will run continuously');
        return;
    }

    // Create observer with 50px margin (start slightly before visible)
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            const carousel = bootstrap.Carousel.getInstance(sliderElement);

            if (!carousel) {
                console.warn('⚠️ Bootstrap Carousel instance not found');
                return;
            }

            if (entry.isIntersecting) {
                // Slider is visible - resume cycling
                //     console.log('👁️ Slider visible - resuming');
                carousel.cycle();
            } else {
                // Slider outside viewport - pause cycling
                //     console.log('🙈 Slider hidden - pausing');
                carousel.pause();
            }
        });
    }, {
        rootMargin: '50px', // Start observing 50px before entering viewport
        threshold: 0.1 // Trigger when 10% of slider is visible
    });

    // Start observing
    observer.observe(sliderElement);

    // console.log('✅ Slider lazy loading active');
}

/**
 * Loads and renders dashboard overview with statistics and charts
 * Displays service status, calendar view, recent checkups, and failures
 * @async
 * @param {string} checkup_period - Time period for dashboard data
 * @param {string} checkup_name - Display name for the period
 * @returns {Promise<void>}
 */
async function load_dashboard_list(checkup_period, checkup_name){
    const block_dashboard_last_checkup = getElById('block_dashboard_last_checkup');
    const tbl_dashboard_last_checkup_list = getElById('tbl_dashboard_last_checkup_list');
    const block_slider = getElById('block_slider');
    const block_slider_btn = getElById('block_slider_btn');
    const graph_slider = getElById('slide_container');
    const block_dashboard_last_failures = getElById('block_dashboard_last_failures');
    const block_dashboard_no_failures = getElById('block_dashboard_no_failures');
    const tbl_dashboard_last_failure_list = getElById('tbl_dashboard_last_failure_list');
    const tbl_dashboard_available_from = getElById('block_dashboard_available_from');

    // Check if calendar already exists
    const isCalendarUpdate = window.calendar_state?.dashboard_calendar;

    graph_slider.innerHTML = '';

    const dashboard_details_data = await manage_api_data('checkupdetails/' + checkup_period, 'get');

    if (dashboard_details_data?.data) {
        const checkup_dashboard_slider_item = await fetch_template('tbl_column/template_slider_item.html');
        const dashboard_attributes = dashboard_details_data.data.attributes;
        const last_checkup_list = dashboard_attributes?.lastCheckups;

        getElById('summary').innerHTML = '';
        if (last_checkup_list.length > 0) {
            const allOk = last_checkup_list.filter(e => e?.statusInfo?.short == 'OK').length;
            const allWarn = last_checkup_list.filter(e => e?.statusInfo?.short == 'Warning').length;
            const allFail = last_checkup_list.filter(e => e?.statusInfo?.short == 'Failure').length;

            // Group services by status
            const healthyServices = last_checkup_list.filter(e => e?.statusInfo?.short == 'OK');
            const warningServices = last_checkup_list.filter(e => e?.statusInfo?.short == 'Warning');
            const offlineServices = last_checkup_list.filter(e => e?.statusInfo?.short == 'Failure');

            const statHealthy = getElById('stat-healthy');
            const statWarning = getElById('stat-warning');
            const statOffline = getElById('stat-offline');

            statHealthy.textContent = allOk;
            statWarning.textContent = allWarn;
            statOffline.textContent = allFail;

            // Update tooltips (title attribute)
            if (healthyServices.length > 0) {
                const names = healthyServices.slice(0, 10).map(s => s.name).join('\n');
                statHealthy.title = healthyServices.length > 10 ? names + '\n...' : names;
            } else {
                statHealthy.title = '';
            }

            if (warningServices.length > 0) {
                const names = warningServices.slice(0, 10).map(s => s.name).join('\n');
                statWarning.title = warningServices.length > 10 ? names + '\n...' : names;
            } else {
                statWarning.title = '';
            }

            if (offlineServices.length > 0) {
                const names = offlineServices.slice(0, 10).map(s => s.name).join('\n');
                statOffline.title = offlineServices.length > 10 ? names + '\n...' : names;
            } else {
                statOffline.title = '';
            }

            // Click handler for Warning (navigate to first problematic service)
            statWarning.style.cursor = warningServices.length > 0 ? 'pointer' : 'default';
            statWarning.onclick = warningServices.length > 0 ? function() {
                const firstWarning = warningServices[0];
                window.location.href = `admin.php?page=checkup_results_menu&checkup_id=${firstWarning.checkupId}&period=${checkup_period}`;
            } : null;

            // Click handler for Offline (navigate to first problematic service)
            statOffline.style.cursor = offlineServices.length > 0 ? 'pointer' : 'default';
            statOffline.onclick = offlineServices.length > 0 ? function() {
                const firstOffline = offlineServices[0];
                window.location.href = `admin.php?page=checkup_results_menu&checkup_id=${firstOffline.checkupId}&period=${checkup_period}`;
            } : null;

            // Find most recent lastCheckupPerformed
            let mostRecentCheckup = null;
            let mostRecentTime = 0;
            last_checkup_list.forEach(checkup => {
                if (checkup.lastCheckupPerformed) {
                    const time = new Date(checkup.lastCheckupPerformed).getTime();
                    if (!isNaN(time) && time > mostRecentTime) {
                        mostRecentTime = time;
                        mostRecentCheckup = checkup.lastCheckupPerformed;
                    }
                }
            });

            // Update last updated time
            const lastUpdatedEl = getElById('stat-last-updated');
            lastUpdatedEl.textContent = mostRecentCheckup ? format_date(mostRecentCheckup, true) : '-';

            makePieChart('summary','overview','Total: ' + last_checkup_list.length, allOk, allWarn, allFail);
            setClass(getElById('summaryChart'), 'd-none');

            // Initialize or update calendar
            if (isCalendarUpdate) {
                await updateCalendarTimespan('dashboard_calendar', last_checkup_list, checkup_period);
            } else {
                await initCalendar('dashboard_calendar', last_checkup_list, checkup_period);
            }
            // Show calendar card
            setClass(getElById('servicesCalendar'), 'd-none');
        }

        const last_checkup_failed_list = dashboard_attributes?.failedList;

        if (last_checkup_list.length > 0) {
            // Build graph slider
            last_checkup_list.forEach((checkup, index) => {
                const tmp_id = checkup.checkupId.toString();
                graph_slider.innerHTML += checkup_dashboard_slider_item.replaceAll('{{checkup_name}}', tmp_id);
                // Dashboard Slider: showAll=false (no doughnut), disableGranularityControls=true, hideChartTypeSelector=true
                showGraphic(parseInt(tmp_id), checkup_period, 'webcheckup-graphic-' + tmp_id, false, null, true, true);
            });

            // Render table
            await render_table(
                'tbl_dashboard_last_checkup_list',
                last_checkup_list,
                'tbl_column/template_tbl_column_last_checkup.html',
                (template, checkup, index) => {
                    const tmp_id = checkup.checkupId.toString();
                    const tmp_last_checkup_item_id = generate_unique_id();

                    return template
                        .replace('{{item_index}}', (index + 1))
                        .replace('{{item_id}}', tmp_last_checkup_item_id)
                        .replace('{{item_uid}}', tmp_id)
                        .replace('{{item_date}}', format_date(checkup.lastCheckupPerformed, false))
                        .replace('{{item_name}}', checkup.name)
                        .replace('{{item_link}}', checkup.link)
                        .replaceAll('{{item_last_message}}', checkup?.statusInfo?.desc ?? '{{item_online_since}}')
                        .replaceAll('{{item_online_since}}', 'Online Since: ' + format_date(checkup.onlineSince))
                        .replace('{{item_result}}', checkup.lastCheckupDescription)
                        .replace('{{item_uptime_percent}}',
                            format_num(checkup.upTimePercent, 2).toString().replace('.00', '').replace(',00', '') + ' %')
                        .replace('{{item_checks_summary}}', 'Completed checkups: ' +
                            (checkup.checkupsOk > 0 ? formatNumbers(checkup.checkupsOk, 0) + ' successful, ' : '') +
                            (formatNumbers(checkup.checkupsFailed, 0) + ' failed'))
                        .replace('{{item_status}}', checkup?.statusInfo?.short === 'OK' ?
                            'tick_mark' : checkup?.statusInfo?.short === 'Warning' ? 'warn_mark' : 'close_mark');
                }
            );

            // Add click handler for Last Website Checkups table rows
            const checkup_rows = tbl_dashboard_last_checkup_list.querySelectorAll('tbody tr');
            checkup_rows.forEach(row => {
                row.style.cursor = 'pointer';
                row.addEventListener('click', function() {
                    const checkup_id = this.getAttribute('data-item_uid');

                    window.location.href = `admin.php?page=checkup_results_menu&checkup_id=${checkup_id}&period=${checkup_period}`;
                });
            });
            
            if (last_checkup_list.length <= 1) {
                setClass(block_slider_btn, '', 'd-none');
            } else {
                setClass(block_slider_btn, 'd-none');
                block_slider.querySelector('.card-header').innerHTML = 'Website Checkups ' + checkup_name;
            }

            const items = getAllElements('.carousel .carousel-item');
            setClass(items[0], '', 'active');

            // Initialize lazy loading for slider after carousel is ready
            setTimeout(() => {
                initSliderLazyLoading();
                // Force chart resize after carousel is visible
                setTimeout(() => {
                    Object.values(Chart.instances).forEach(chart => {
                        if (chart && chart.canvas && chart.canvas.offsetParent !== null) {
                            chart.resize();
                        }
                    });
                }, 200);
            }, 100);
        } else {
            setClass(block_dashboard_last_checkup, '', 'd-none');
            setClass(block_slider, '', 'd-none');
        }

        if (last_checkup_failed_list && last_checkup_failed_list.length > 0) {
            // Flatten data
            const flat_errors = [];
            let tmp_counter = 0;

            last_checkup_failed_list.forEach((error_data) => {
                if (error_data.length) {
                    error_data.forEach((last_checkup) => {
                        flat_errors.push(last_checkup);
                    });
                } else {
                    tmp_counter++;
                    flat_errors.push({...error_data, _counter: tmp_counter});
                }
            });
            await render_table(
                'tbl_dashboard_last_failure_list',
                flat_errors,
                'tbl_column/template_tbl_column_last_failure.html',
                (template, error_data, index) => {
                    const tmp_item_id = generate_unique_id();
                    const counter = error_data._counter || (index + 1);
                    const times = error_data.checkups + ' time' + (parseInt(error_data.checkups) > 1 ? 's' : '');

                    return template
                        .replace('{{item_index}}', counter.toString())
                        .replace('{{item_id}}', tmp_item_id)
                        .replace('{{item_uid}}', error_data.checkupId.toString())
                        .replace('{{item_times}}', times)
                        .replace('{{item_date}}', getTimespan(error_data.from, error_data.until, false))
                        .replace('{{item_name}}', error_data.checkup)
                        .replace('{{item_url}}', error_data.link)
                        .replace('{{item_result}}', error_data.description);
                }
            );

            // Add click handler for Recent Failures table rows
            const failure_rows = tbl_dashboard_last_failure_list.querySelectorAll('tbody tr');
            failure_rows.forEach(row => {
                row.style.cursor = 'pointer';
                row.addEventListener('click', function() {
                    const checkup_id = this.getAttribute('data-uid');
                    window.location.href = `admin.php?page=checkup_results_menu&checkup_id=${checkup_id}&period=${checkup_period}`;
                });
            });

            setClass(block_dashboard_last_failures, 'd-none');
            setClass(block_dashboard_no_failures, '', 'd-none');
        } else {
            setClass(block_dashboard_last_failures, '', 'd-none');
            setClass(block_dashboard_no_failures, 'd-none');
        }

        if (dashboard_attributes?.firstExecutionPlanned) {
            tbl_dashboard_available_from.innerHTML = dashboard_attributes.firstExecutionPlanned;
            setClass(tbl_dashboard_available_from, 'd-none');
        }
    } else {
        generate_multiple_toast(dashboard_details_data?.errors);
        close_loader_safe();
        setClass(getElById('block_dashboard'), '', 'd-none');
    }
    close_loader_safe();
}

/* PAGE FUNCTION */

/**
* Main initialization on DOM ready
* Routes to appropriate page handler based on current WordPress admin page
*/
document.addEventListener('DOMContentLoaded', async function () {
    if (document.body.classList.contains('website_checkups')) {
        wc_site_url = website_checkups_code['site_url'];
        wc_api_loc = website_checkups_code['api_loc'];
        wc_nonce = website_checkups_code['nonce'];
        wc_template_loc = website_checkups_code['template_loc'];
        wc_user_data = website_checkups_code['user_data'];
        wc_page_class = website_checkups_code['page_class'];

        // Load field definitions from wp_options
        const post_data = {
            'post_type': 'load_form'
        };
        await manage_local_data(post_data);

        AuthTokenWebsiteCheckups = wc_user_data['token'];
        page_title = getElById('page_title');
        page_container = getElById('page_container');

        await check_main_menu();

        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('hide.bs.modal', function(event) {
                // Remove focus before Bootstrap sets aria-hidden
                const activeElement = document.activeElement;
                if (activeElement && this.contains(activeElement)) {
                    activeElement.blur();
                }
            });
        });

        if (wc_page_class === 'toplevel_page_website_checkups_menu') {

            if (!wc_user_data['token']) {
                setTimeout(async function () {
                    await registration_page();
                    close_loader_safe();
                }, 750);
                page_title.innerText = '';

            } else {
                await setting_page();
                page_title.innerText = 'Settings';
            }
            close_loader_safe();
        }
        if (wc_page_class === 'website-checkups_page_notifications_menu') {
            page_title.innerText = 'Notifications';
            modal_name = 'notifications';
            await notification_page();

            const info_tooltip_list = getAllElements('.info_tooltip');
            info_tooltip_list.forEach(tooltip_btn => {
                tooltip_btn.addEventListener('click', async function (e) {
                    e.preventDefault();
                    if (this.getAttribute('data-info') !== 'null') {
                        show_toast('light', this.getAttribute('data-info'));
                    }
                });
            });
            close_loader_safe();
        }
        if (wc_page_class === 'website-checkups_page_checkups_menu') {
            page_title.innerText = 'Checkups';
            modal_name = 'sitecheckups';
            await checkup_page();
            close_loader_safe();
        }
        if (wc_page_class === 'website-checkups_page_checkup_results_menu') {
            page_title.innerText = 'Checkup Results';
            await checkup_details_page();
            close_loader_safe();
        }
        if (wc_page_class === 'website-checkups_page_dashboard_menu') {
            page_title.innerText = 'Dashboard';
            await dashboard_page();
            close_loader_safe();
        }
    }
});
