/* /lib/js/api.js */
/*!
 * API & Caching Layer for Website Checkups Plugin v1.2.0
 * https://website-checkups.com
 * Extracted from admin.js during refactoring
 * Released under the GPLv2 license
 */

// ===================================
// CONSTANTS & CONFIGURATION
// ===================================

let cached_endpoints = ['countries', 'countries?filter[popular]=1', 'languages', 'languages?filter[popular]=1',
    'subproducts', 'fielddescriptions'];
//let tmp_cached_endpoints = ['users', 'subscriptions', 'registrations', 'notifications', 'invoices'];

// Cache configuration
const CONSISTENCY_CACHE_SECONDS = 1;
const MIN_DYNAMIC_CACHE_SECONDS = 5;
const PUBLIC_ENDPOINTS_TTL = 24 * 60 * 60;

// Cache TTL Configuration for invalidatable endpoints (in seconds)
const CACHE_TTL_CONFIG = {
    'sitecheckups': 30 * 60,  // 30 minutes
    'notifications': 30 * 60, // 30 minutes
    'subscriptions': 30 * 60, // 30 minutes
    'invoices': 30 * 60,      // 30 minutes
    'registrations': 30 * 60  // 30 minutes
};

const CACHE_INVALIDATION_PREFIX = 'webcheckups_cache_invalidation:';

// Request Deduplication - prevents parallel identical requests
const pendingRequests = new Map();

// Cache invalidation - stores timestamps of last changes
const cacheInvalidationTimestamps = new Map();

// ========================
// LOCALSTORAGE FUNCTIONS
// ========================

/**
 * Retrieves cached data from localStorage
 *
 * @param {string} key - Cache key identifier
 * @returns {*|null} Cached data or null if not found/expired
 *
 * @description
 * Checks localStorage for cached data with 24-hour TTL.
 * Invalidates cache if plugin version changed.
 */
function get_webcheckups_storage_data(key)
{
    if (localStorage) {
        const time = Math.ceil((new Date()).getTime()/1000);
        const data_string = localStorage.getItem('webcheckups:' + key);
        const data_obj = data_string ? JSON.parse(data_string) : null;
        const time_diff = data_obj?.saved ? time - data_obj?.saved : 9999999;

        // Skip for day-level cache (handled separately)
        if (key.startsWith('stats:')) {
            return null; // Day-level cache uses its own logic
        }

        // Invalidate cache on plugin update
        if (key === 'form_fields_data' && data_obj?.version !== wc_site_url) {
            // console.log('🔄 Cache outdated, clearing...');
            localStorage.removeItem('webcheckups:' + key);
            return null;
        }

        if (data_obj?.data && time_diff < 24 * 60 * 60) {
            return data_obj?.data ?? null;
        }
    }
    return null;
}

/**
 * Stores data in localStorage cache
 *
 * @param {string} key - Cache key identifier
 * @param {*} data - Data to cache
 * @returns {void}
 *
 * @description
 * Saves data with timestamp and plugin version for cache validation.
 */
function set_webcheckups_storage_data(key, data)
{
    if (localStorage) {
        const time = Math.ceil((new Date()).getTime()/1000);
        const cache_obj = {
            'saved': time,
            'version': website_checkups_code['plugin_version'],
            'data': data
        };
        localStorage.setItem('webcheckups:' + key, JSON.stringify(cache_obj));
    }
}

/**
 * Clears old cache entries when quota is exceeded
 * Removes oldest 30% of cached items
 */
function clear_old_cache_entries() {
    const entries = [];

    // Collect all webcheckups cache entries
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith('webcheckups:')) {
            try {
                const data = JSON.parse(localStorage.getItem(key));
                entries.push({
                    key: key,
                    timestamp: data.timestamp || 0
                });
            } catch (e) {
                // Invalid entry, mark for removal
                entries.push({
                    key: key,
                    timestamp: 0
                });
            }
        }
    }

    // Sort by timestamp (oldest first)
    entries.sort((a, b) => a.timestamp - b.timestamp);

    // Remove oldest 30%
    const removeCount = Math.ceil(entries.length * 0.3);
    for (let i = 0; i < removeCount && i < entries.length; i++) {
        localStorage.removeItem(entries[i].key);
        // console.log('🗑️ Removed old cache:', entries[i].key.substring(0, 80) + '...');
    }

    // console.log(`✅ Cleared ${removeCount} old cache entries`);
}

/**
 * Shows loader with automatic timeout
 *
 * @param {boolean} [transparent_bg=true] - Whether to use transparent background
 * @param {string} [loader_id='page_loader'] - Loader element ID
 * @param {string} [loader_location='.loader_block'] - Loader container selector
 * @param {number} [timeout_ms=50000] - Timeout in milliseconds
 * @returns {void}
 *
 * @description
 * Displays loader and automatically closes it after timeout to prevent stuck UI.
 */
function show_loader_with_timeout(transparent_bg = true, loader_id = 'page_loader', loader_location = '.loader_block', timeout_ms = 50000) {
    show_loader(transparent_bg, loader_id, loader_location);

    // Clear old timeout if exists
    if (loader_timeout_id) {
        clearTimeout(loader_timeout_id);
    }

    // Set new timeout
    loader_timeout_id = setTimeout(() => {
        console.warn('⚠️  Loader timeout - force closing');
        close_loader(loader_id);
        show_toast(false, 'Operation took too long. Please try again.', 5);
    }, timeout_ms);
}

/**
 * Safely closes loader and clears timeout
 *
 * @param {string} [loader_id='page_loader'] - Loader element ID
 * @returns {void}
 *
 * @description
 * Ensures loader is properly closed and timeout is cleared to prevent memory leaks.
 */
function close_loader_safe(loader_id = 'page_loader') {
    // Clear timeout
    if (loader_timeout_id) {
        clearTimeout(loader_timeout_id);
        loader_timeout_id = null;
    }
    close_loader(loader_id);
}

/**
 * Checks if endpoint is public (no authentication required)
 *
 * @param {string} url - API endpoint URL
 * @returns {boolean} True if endpoint is public
 *
 * @description
 * Public endpoints: subproducts, languages, countries, fielddescriptions
 */
function is_public_endpoint(url) {
    const public_endpoints = ['subproducts', 'languages', 'countries', 'fielddescriptions'];
    const base_url = url.split('?')[0];
    return public_endpoints.some(endpoint =>
        base_url.includes('/' + endpoint) || base_url.endsWith(endpoint)
    );
}

/**
 * Checks if endpoint supports dynamic caching
 *
 * @param {string} url - API endpoint URL
 * @returns {boolean} True if endpoint supports dynamic TTL from response
 *
 * @description
 * Dynamic cache endpoints: statistics, checkupdetails
 * These endpoints return cache TTL in response meta.stat.cache
 */
function supports_dynamic_cache(url) {
    const base_url = url.split('?')[0];
    return base_url.includes('statistics') || base_url.includes('checkupdetails');
}

/**
 * Extracts dynamic TTL from API response
 *
 * @param {Object} response_data - API response object
 * @returns {number|null} TTL in seconds or null if not available
 *
 * @description
 * Reads cache TTL from response.meta.stat.cache:
 * - 0: Resource being checked → use consistency cache (1 second)
* - >= 5: Valid TTL → use as-is
* - 1-4: Too short → use consistency cache (1 second)
* - null/undefined: No dynamic caching
*/
function extract_dynamic_ttl(response_data) {
    try {
        const cache_ttl = response_data?.meta?.stat?.cache;

        // cache: 0 means "currently being checked" → use consistency cache
        if (cache_ttl === 0) {
            return CONSISTENCY_CACHE_SECONDS;
        }

        // Valid TTL found (>= MIN_DYNAMIC_CACHE_SECONDS)
        if (typeof cache_ttl === 'number' && cache_ttl >= MIN_DYNAMIC_CACHE_SECONDS) {
            return cache_ttl;
        }

        // TTL too short (but > 0) → also use consistency cache
        if (typeof cache_ttl === 'number' && cache_ttl > 0 && cache_ttl < MIN_DYNAMIC_CACHE_SECONDS) {
            return CONSISTENCY_CACHE_SECONDS;
        }

        return null;
    } catch (error) {
        return null;
    }
}

/**
* Determines if consistency cache should be used
*
* @param {string} url - API endpoint URL
* @returns {boolean} True if consistency cache (1 second TTL) should be used
*
* @description
* Uses consistency cache for private GET requests that don't have:
* - Public endpoint status
* - Dynamic caching support
* - Static caching configuration
*/
function should_use_consistency_cache(url) {
    // Not for public endpoints
    if (is_public_endpoint(url)) return false;

    // Not for dynamic cache endpoints
    if (supports_dynamic_cache(url)) return false;

    // Not for already statically cached endpoints
    const base_url = url.split('?')[0];
    if (cached_endpoints.some(endpoint => base_url.includes(endpoint))) return false;

    // For all other private GET requests
    return true;
}

/**
* Gets cache configuration for endpoint
*
* @param {string} api_end_point - API endpoint path
* @returns {Object} Configuration object: {ttl: number, invalidation_key: string|null}
*
* @description
* Returns specific cache TTL and invalidation key for configured endpoints.
* Falls back to consistency cache for unconfigured endpoints.
*/
function get_endpoint_cache_config(api_end_point) {
    // Check if endpoint has specific cache configuration
    for (const [key, ttl] of Object.entries(CACHE_TTL_CONFIG)) {
        if (api_end_point.includes(key)) {
            return { ttl: ttl, invalidation_key: key };
        }
    }

    // Fallback to consistency cache for other private endpoints
    return { ttl: CONSISTENCY_CACHE_SECONDS, invalidation_key: null };
}

/**
* Generates cache key for request deduplication
*
* @param {string} method - HTTP method (GET, POST, etc.)
* @param {string} url - API endpoint URL
* @param {Object|null} params - Request parameters
* @returns {string} Unique cache key
*
* @description
* Creates unique key combining method, URL, and parameters.
* Used to prevent duplicate simultaneous requests.
*/
function generate_cache_key(method, url, params) {
    let key = `${method}:${url}`;
    if (params && Object.keys(params).length > 0) {
        key += ':' + JSON.stringify(params);
    }
    return key;
}

/**
* Retrieves cached data with custom TTL support
*
* @param {string} key - Cache key identifier
* @param {number|null} [custom_ttl=null] - Custom TTL in seconds, uses stored TTL if null
* @returns {*|null} Cached data or null if not found/expired/invalidated
*
* @description
* Enhanced cache retrieval with:
* - Custom or stored TTL
* - Plugin version validation
* - Cache invalidation timestamp checking
*/
function get_webcheckups_storage_data_with_ttl(key, custom_ttl = null) {
    if (!localStorage) return null;

    const time = Math.ceil((new Date()).getTime() / 1000);
    const data_string = localStorage.getItem('webcheckups:' + key);
    const data_obj = data_string ? JSON.parse(data_string) : null;

    if (!data_obj?.data) return null;

    const time_diff = data_obj?.saved ? time - data_obj.saved : 9999999;
    const ttl = custom_ttl || data_obj?.ttl || (24 * 60 * 60);

    // Invalidate cache on plugin update
    if (data_obj?.version !== website_checkups_code['plugin_version']) {
        localStorage.removeItem('webcheckups:' + key);
        return null;
    }

    // Check cache invalidation
    const invalidation_key = data_obj?.invalidation_key;
    if (invalidation_key && cacheInvalidationTimestamps.has(invalidation_key)) {
        const invalidation_time = cacheInvalidationTimestamps.get(invalidation_key);
        if (invalidation_time > data_obj.saved) {
            localStorage.removeItem('webcheckups:' + key);
            return null;
        }
    }

    if (time_diff < ttl) {
        return data_obj.data;
    }

    return null;
}

/**
* Stores data in cache with TTL and invalidation key
*
* @param {string} key - Cache key identifier
* @param {*} data - Data to cache
* @param {number|null} [ttl=null] - Time-to-live in seconds, defaults to 24 hours
* @param {string|null} [invalidation_key=null] - Key for event-based cache invalidation
* @returns {void}
*
* @description
* Enhanced cache storage with configurable TTL and invalidation support.
* Stores timestamp, plugin version, TTL, and optional invalidation key.
*/
function set_webcheckups_storage_data_with_ttl(key, data, ttl = null, invalidation_key = null) {
    if (!localStorage) return;

    const time = Math.ceil((new Date()).getTime() / 1000);
    const cache_obj = {
        'saved': time,
        'version': website_checkups_code['plugin_version'],
        'ttl': ttl || (24 * 60 * 60),
        'data': data
    };

    if (invalidation_key) {
        cache_obj.invalidation_key = invalidation_key;
    }

    try {
        localStorage.setItem('webcheckups:' + key, JSON.stringify(cache_obj));
    } catch (e) {
        if (e.name === 'QuotaExceededError') {
            console.warn('⚠️ localStorage quota exceeded. Clearing old cache...');
            clear_old_cache_entries();

            // Retry after cleanup
            try {
                localStorage.setItem('webcheckups:' + key, JSON.stringify(cache_obj));
            } catch (retryError) {
                console.error('❌ Still cannot cache after cleanup:', retryError);
                // Continue without caching
            }
        } else {
            throw e;
        }
    }
}

/**
* Invalidates cache for specific endpoint group
*
* @param {string} invalidation_key - Invalidation key (e.g., 'subscriptions', 'notifications')
* @returns {void}
*
* @description
* Stores current timestamp for invalidation key.
* All cached items with this key will be invalidated on next access.
*/
function invalidate_cache(invalidation_key) {
    const timestamp = Math.ceil((new Date()).getTime() / 1000);
    cacheInvalidationTimestamps.set(invalidation_key, timestamp);

    // Store in localStorage for persistence across sessions
    set_cache_invalidation_timestamp(invalidation_key, timestamp);

    // console.log(`🗑️ Cache invalidated: ${invalidation_key} at ${new Date(timestamp * 1000).toISOString()}`);
}

/**
 * Get cache invalidation timestamp from localStorage
 * @param {string} invalidation_key - Invalidation key (e.g., "checkup:123")
 * @returns {number|null} Timestamp or null if not found
 */
function get_cache_invalidation_timestamp(invalidation_key) {
    const storageKey = CACHE_INVALIDATION_PREFIX + invalidation_key;
    const timestamp = localStorage.getItem(storageKey);
    return timestamp ? parseInt(timestamp) : null;
}

/**
 * Set cache invalidation timestamp in localStorage
 * @param {string} invalidation_key - Invalidation key (e.g., "checkup:123")
 * @param {number} timestamp - Unix timestamp
 */
function set_cache_invalidation_timestamp(invalidation_key, timestamp) {
    const storageKey = CACHE_INVALIDATION_PREFIX + invalidation_key;
    localStorage.setItem(storageKey, timestamp.toString());
}

/**
 * Get localStorage usage statistics
 * @returns {Object} Usage stats {used_bytes, used_mb, limit_mb, percentage, items_count}
 */
function get_localstorage_usage() {
    let totalBytes = 0;
    let itemsCount = 0;

    for (let key in localStorage) {
        if (localStorage.hasOwnProperty(key)) {
            // Count key + value size in bytes (UTF-16, so 2 bytes per char)
            totalBytes += (key.length + localStorage[key].length) * 2;
            itemsCount++;
        }
    }

    const usedMB = (totalBytes / (1024 * 1024)).toFixed(2);
    const limitMB = 5; // Most browsers have 5-10MB limit, we assume 5MB
    const percentage = ((totalBytes / (limitMB * 1024 * 1024)) * 100).toFixed(1);

    return {
        used_bytes: totalBytes,
        used_mb: usedMB,
        limit_mb: limitMB,
        percentage: parseFloat(percentage),
        items_count: itemsCount,
        formatted: `${usedMB} MB / ${limitMB} MB (${percentage}%)`
    };
}

/**
 * Log localStorage usage to console
 */
function log_localstorage_usage() {
    const usage = get_localstorage_usage();

    console.log('📊 localStorage Usage:');
    console.log(`  Total: ${usage.formatted}`);
    console.log(`  Items: ${usage.items_count}`);
    console.log(`  Bytes: ${usage.used_bytes.toLocaleString()}`);

    if (usage.percentage > 80) {
        console.warn('⚠️ localStorage is over 80% full!');
    } else if (usage.percentage > 90) {
        console.error('🚨 localStorage is over 90% full!');
    }

    return usage;
}

/**
 * Show all cache invalidation timestamps (DEBUG)
 */
function debug_cache_invalidations() {
    console.log('🔍 Cache Invalidation Timestamps:');

    // From localStorage
    const localStorageKeys = Object.keys(localStorage)
        .filter(key => key.startsWith(CACHE_INVALIDATION_PREFIX));

    console.log(`  In localStorage: ${localStorageKeys.length} entries`);
    localStorageKeys.forEach(key => {
        const invalidationKey = key.replace(CACHE_INVALIDATION_PREFIX, '');
        const timestamp = localStorage.getItem(key);
        const date = new Date(parseInt(timestamp) * 1000);
        console.log(`    ${invalidationKey}: ${date.toISOString()}`);
    });

    // From memory (session)
    console.log(`  In memory (session): ${cacheInvalidationTimestamps.size} entries`);
    cacheInvalidationTimestamps.forEach((timestamp, key) => {
        const date = new Date(timestamp * 1000);
        console.log(`    ${key}: ${date.toISOString()}`);
    });
}
// ===================================
// API REQUEST FUNCTIONS
// ===================================

/**
* Executes actual API request (without caching logic)
*
* @async
* @param {string} api_end_point - API endpoint path (without base URL)
* @param {string} api_method - HTTP method ('get', 'post', 'patch', 'delete')
* @param {Object|null} [form_data=null] - Request data/parameters
* @returns {Promise<Object>} API response data or error object
*
* @description
* Low-level API request handler called by manage_api_data.
* Features:
* - 50 second timeout
* - Bearer token authentication
* - Comprehensive error handling (401, 403, 404, 500+)
* - User-friendly error messages via toast notifications
* - Automatic loader management
*/
async function execute_api_request(api_end_point, api_method, form_data = null) {
    show_loader_with_timeout();
    let url = wc_api_loc + api_end_point;

    let headers = {
        'Content-Type': 'application/vnd.api+json', 'Accept': 'application/vnd.api+json',
    };

    if (wc_user_data['token']) {
        headers['Authorization'] = 'Bearer ' + wc_user_data['token'];
    }

    let options = {
        method: api_method.toUpperCase(),
        headers: headers,
        signal: AbortSignal.timeout(50000),
    };

    let input_data = {
        data: {
            type: api_end_point,
        },
    };

    if (form_data !== null) {
        if (api_method !== 'get' && api_method !== 'post') {
            if (form_data?.item_id) {
                url += '/' + form_data.item_id;
                input_data.data.id = form_data.item_id;
                delete form_data.item_id;
            }
        }

        if (api_method === 'post') {
            if (form_data?.bulk_select) {
                url = wc_api_loc + 'bulk/' + api_end_point + '/' + form_data.bulk_type;
                delete form_data.bulk_select;
                delete form_data.bulk_type;
            }
        }

        input_data.data.attributes = form_data;
        options.body = JSON.stringify(input_data);
    }

    try {
        const response = await fetch(url, options);
        // ===================================
        // JSON API SUCCESS CODES (VOR Fehlerbehandlung!)
        // ===================================

        // 204 No Content - Erfolg ohne Body (typisch bei DELETE)
        if (response.status === 204) {
            close_loader_safe();
            return {
                meta: {
                    notice: 'Successfully deleted'
                }
            };
        }

        // 206 Partial Content - JSON API Standard für partielle Daten
        if (response.status === 206) {
            close_loader_safe();
            const responseData = await response.json();
            return responseData;
        }

        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error:', {
                status: response.status,
                statusText: response.statusText,
                url: url,
                method: api_method,
                response: errorText
            });

            close_loader_safe();

            if (response.status === 401) {
                show_toast(false, 'Authentication failed. Please log in again.', 5);
                return { errors: [{ detail: 'Unauthorized' }] };
            } else if (response.status === 403) {
                show_toast(false, 'Access denied. Insufficient permissions.', 5);
                return { errors: [{ detail: 'Forbidden' }] };
            } else if (response.status === 404) {
                show_toast(false, 'Resource not found. Please try again.', 5);
                return { errors: [{ detail: 'Not found' }] };
            } else if (response.status === 422) {
                // Validation errors - parse JSON and return errors array
                // DON'T show toast here - handle_toast will call generate_multiple_toast
                try {
                    const errorData = JSON.parse(errorText);

                    // Return errors in the format expected by generate_multiple_toast
                    if (errorData?.errors && Array.isArray(errorData.errors)) {
                        console.log('🔴 Validation errors:', errorData.errors);
                        return { errors: errorData.errors };
                    }

                    // Fallback if errors not in expected format
                    return { errors: [{ detail: 'Validation failed. Please check your input.' }] };
                } catch (parseError) {
                    console.error('Failed to parse 422 error response:', parseError);
                    return { errors: [{ detail: 'Validation failed. Please check your input.' }] };
                }
            } else if (response.status >= 500) {
                show_toast(false, 'Server error. Please try again later.', 5);
                return { errors: [{ detail: 'Server error' }] };
            } else {
                show_toast(false, `Request failed: ${response.statusText}`, 5);
                return { errors: [{ detail: response.statusText }] };
            }
        }

        const responseData = await response.json();

        return responseData;
    } catch (error) {
        close_loader_safe();

        console.error('API Request Failed:', {
            error: error,
            url: url,
            method: api_method,
            endpoint: api_end_point
        });

        if (error.name === 'TimeoutError' || error.name === 'AbortError') {
            show_toast(false, 'Request timeout. Please check your internet connection.', 5);
        } else if (error instanceof TypeError && error.message.includes('fetch')) {
            show_toast(false, 'Network error. Please check your internet connection.', 5);
        } else if (error instanceof SyntaxError) {
            show_toast(false, 'Invalid server response. Please contact support.', 5);
        } else {
            show_toast(false, 'An unexpected error occurred. Please try again.', 5);
        }

        return {
            errors: [{
                detail: error.message || 'Unknown error',
                source: { pointer: api_end_point }
            }]
        };
    }
}

/**
* Main API request function with intelligent caching
*
* @async
* @param {string} api_end_point - API endpoint path
* @param {string} api_method - HTTP method ('get', 'post', 'patch', 'delete')
* @param {Object|null} [form_data=null] - Request data/parameters
* @param {Object} [options={}] - Additional options
* @param {string|null} [options.variantNeeded=null] - Variant filter ('daily', 'hourly', 'realtime')
* @returns {Promise<Object>} API response data or error object
*
* @description
* Intelligent API request manager with multi-tier caching strategy:
*
* **GET Requests - 4 Caching Strategies:**
* 1. Static Cache: Long-term public data (countries, languages, subproducts)
* 2. Public Cache: 24h TTL for public endpoints
* 3. Dynamic Cache: TTL from API response (statistics, checkupdetails)
* 4. Consistency Cache: 1s TTL for other private endpoints
*
* **POST/PATCH/DELETE Requests:**
* - No caching
* - Automatic cache invalidation for related endpoints
* - Special handling: subscription changes also invalidate invoices
*
* **Request Deduplication:**
* Prevents multiple simultaneous identical requests using pending promises.
*
* @example
* // GET with caching
* const data = await manage_api_data('statistics?filter[checkupId]=123', 'get');
*
* // POST with cache invalidation
* await manage_api_data('subscriptions', 'post', {plan: 'premium'});
*/
async function manage_api_data(api_end_point, api_method, form_data = null, options = {})
{
    const variantNeeded = options.variantNeeded || null;

    // GET request caching logic
    if (api_method === 'get') {
        const is_public = is_public_endpoint(api_end_point);
        const is_dynamic = supports_dynamic_cache(api_end_point);
        const use_consistency = should_use_consistency_cache(api_end_point);
        const is_static_cached = cached_endpoints.includes(api_end_point);

        // Generate cache key
        const cache_key = generate_cache_key('GET', api_end_point, form_data);

        // Strategy 1: Static public/long-term endpoints
        if (is_static_cached) {
            const cached = get_webcheckups_storage_data(api_end_point);
            if (cached) {
                return cached;
            }
        }

        // ===================================
        // TIMELINE CACHE FOR STATISTICS
        // ===================================
        // Check if this is a statistics request that should use day-level caching
        const isStatisticsRequest = api_end_point.includes('statistics') &&
            api_end_point.includes('filter[checkupId]=') &&
            api_end_point.includes('filter[fromDate]=');

        if (isStatisticsRequest && typeof load_statistics_with_timeline_cache === 'function') {
            // Extract checkupId and fromDate from endpoint
            const checkupIdMatch = api_end_point.match(/filter\[checkupId\]=(\d+)/);
            const fromDateMatch = api_end_point.match(/filter\[fromDate\]=([^&]+)/);

            if (checkupIdMatch && fromDateMatch) {
                const checkupId = parseInt(checkupIdMatch[1]);
                const fromDate = fromDateMatch[1];

                // Request deduplication: If a request for this checkupId is already running,
                // wait for it instead of starting a new one
                const dedupeKey = `timeline:${checkupId}`;

                if (pendingRequests.has(dedupeKey)) {
                    // console.log(`📊 Timeline request for checkup ${checkupId} already pending, reusing...`);
                    const existingPromise = pendingRequests.get(dedupeKey);
                    // Wait for existing request, but still filter by current fromDate
                    const result = await existingPromise;
                    // Return same result (filtering happens in timeline cache)
                    // console.log(`✅ DEDUPE: Reused result for checkup ${checkupId}`);
                    return result;
                }
                
                // console.log('📊 Using timeline cache for statistics');
                // console.log(`🔍 DEDUPE: Creating new request for key=${dedupeKey}`);

                // CRITICAL: Create and store promise SYNCHRONOUSLY before any await
                // This prevents race condition in Promise.all() scenarios
                const requestPromise = (async () => {
                    try {
                        // Create API fetch function with tillDate support
                        const apiFetch = async (id, from, till = null) => {
                            let endpoint = `statistics?filter[checkupId]=${id}&filter[fromDate]=${from}`;
                            if (till) {
                                endpoint += `&filter[tillDate]=${till}`;
                            }
                            return await execute_api_request(endpoint, 'get', null);
                        };

                        return await load_statistics_with_timeline_cache(checkupId, fromDate, apiFetch, variantNeeded);
                    } finally {
                        // Cleanup after completion
                        pendingRequests.delete(dedupeKey);
                    }
                })();

                // Store IMMEDIATELY - must happen before any await
                pendingRequests.set(dedupeKey, requestPromise);

                return await requestPromise;
            }
        }
        // ===================================

        // Strategy 2: Public endpoints with long TTL (24h)
        if (is_public) {
            const cached = get_webcheckups_storage_data_with_ttl(cache_key, PUBLIC_ENDPOINTS_TTL);
            if (cached) {
                // console.log('📦 Public endpoint loaded from cache:', api_end_point);
                return cached;
            }

            // Execute request
            const responseData = await execute_api_request(api_end_point, api_method, form_data);

            // Save in cache by success
            if (!responseData?.errors) {
                set_webcheckups_storage_data_with_ttl(cache_key, responseData, PUBLIC_ENDPOINTS_TTL);
                // console.log('💾 Public endpoint saved to cache:', api_end_point);
            }

            return responseData;
        }

        // Strategy 3 & 4: Dynamic or consistency caching
        if (is_dynamic || use_consistency) {
            const cached = get_webcheckups_storage_data_with_ttl(cache_key);
            if (cached) {
                return cached;
            }

            // Request deduplication: Check if request is already in progress
            if (pendingRequests.has(cache_key)) {
                return await pendingRequests.get(cache_key);
            }

            // Request deduplication: Create promise for this request
            const request_promise = (async () => {
                try {
                    const responseData = await execute_api_request(api_end_point, api_method, form_data);

                    // Apply caching strategies after successful request
                    if (!responseData?.errors) {
                        // Strategy 3: Dynamic caching (statistics, checkupdetails)
                        if (is_dynamic) {
                            const dynamic_ttl = extract_dynamic_ttl(responseData);
                            if (dynamic_ttl !== null) {
                                set_webcheckups_storage_data_with_ttl(cache_key, responseData, dynamic_ttl);
                            }
                        }
                        // Strategy 4: Configured cache for private endpoints with invalidation
                        else if (use_consistency) {
                            const cache_config = get_endpoint_cache_config(api_end_point);

                            set_webcheckups_storage_data_with_ttl(
                                cache_key,
                                responseData,
                                cache_config.ttl,
                                cache_config.invalidation_key
                            );
                        }
                    }

                    return responseData;
                } finally {
                    // Cleanup: Remove from pending requests after completion
                    pendingRequests.delete(cache_key);
                }
            })();

            // Store promise for deduplication
            pendingRequests.set(cache_key, request_promise);

            return await request_promise;
        }
    }

    // For non-cached GET requests and all POST/PATCH/DELETE requests
    const responseData = await execute_api_request(api_end_point, api_method, form_data);

    // Apply static caching for GET requests to static cached endpoints
    if (api_method === 'get' && cached_endpoints.includes(api_end_point) && !responseData?.errors) {
        set_webcheckups_storage_data(api_end_point, responseData);
    }

    // Cache invalidation for mutation requests (POST, PATCH, DELETE)
    if (['post', 'patch', 'delete'].includes(api_method) && !responseData?.errors) {
        // Invalidate cache for all configured endpoints
        for (const key of Object.keys(CACHE_TTL_CONFIG)) {
            if (api_end_point.includes(key)) {
                invalidate_cache(key);

                // Special case: subscription changes also invalidate invoices
                if (key === 'subscriptions') {
                    invalidate_cache('invoices');
                }
                break;
            }
        }
        // ===================================
        // TIMELINE CACHE INVALIDATION FOR CHECKUPS
        // ===================================
        if (api_end_point.includes('sitecheckups')) {
            // Case 1: Single checkup PATCH or DELETE
            if (api_method === 'patch' || api_method === 'delete') {
                const checkupIdMatch = api_end_point.match(/sitecheckups\/(\d+)/);
                if (checkupIdMatch) {
                    const checkupId = checkupIdMatch[1];

                    if (api_method === 'delete') {
                        // Always invalidate on delete
                        invalidate_cache(`checkup:${checkupId}`);
                        // console.log(`🗑️ Timeline cache invalidated: checkup ${checkupId} (deleted)`);
                    } else if (api_method === 'patch' && form_data) {
                        // Only invalidate for relevant field changes
                        const relevantFields = ['url', 'requestAs', 'port', 'active'];
                        const hasRelevantChange = relevantFields.some(field =>
                            form_data.hasOwnProperty(field)
                        );

                        if (hasRelevantChange) {
                            const changedFields = relevantFields.filter(f => form_data.hasOwnProperty(f));
                            invalidate_cache(`checkup:${checkupId}`);
                            // console.log(`🗑️ Timeline cache invalidated: checkup ${checkupId} (changed: ${changedFields.join(', ')})`);
                        }
                    }
                }
            }

            // Case 2: Bulk operations (deactivate/remove)
            // Note: Bulk URL is 'bulk/sitecheckups/{bulk_type}'
            if (api_method === 'post' && api_end_point.includes('bulk/sitecheckups')) {
                // Extract bulk_type from endpoint (already in URL at this point)
                const bulkTypeMatch = api_end_point.match(/bulk\/sitecheckups\/(\w+)/);
                const bulkType = bulkTypeMatch ? bulkTypeMatch[1] : null;

                // form_data.checkupId contains array of IDs
                if (form_data?.checkupId && Array.isArray(form_data.checkupId)) {
                    const checkupIds = form_data.checkupId;

                    if (bulkType === 'deactivate' || bulkType === 'remove') {
                        checkupIds.forEach(id => {
                            invalidate_cache(`checkup:${id}`);
                        });
                        // console.log(`🗑️ Bulk ${bulkType}: Invalidated ${checkupIds.length} checkup timeline caches`);
                    }
                }
            }
        }
        // ===================================        
    }

    return responseData;
}

// ===================================
// DATA CHECKING & MENU FUNCTIONS
// ===================================

/**
* Checks if data exists in cache or API
*
* @async
* @param {string} type - Data type identifier (e.g., 'registrations', 'invoices')
* @returns {Promise<boolean>} True if data exists
*
* @description
* Checks localStorage cache first, then queries API if not cached.
* For registrations and invoices: checks if at least one entry exists.
* Caches existence status to avoid repeated API calls.
*/
async function check_data_exists(type) {
    const partKey = wc_user_data?.token ? wc_user_data.token.substring(5, 20) : '';
    const key = type.substring(0, 3).toUpperCase() + partKey;

    const data = get_webcheckups_storage_data(key) ?? await manage_api_data(type, 'get');
    if (['registrations', 'invoices'].includes(type)) {
        if (data?.entryexists) {
            return true;
        } else if (!data?.data || data.data?.errors || data.data.length < 1 ) {
            return false;
        } else if (data?.data?.attributes || (data?.data && data.data.length >= 1)) {
            set_webcheckups_storage_data(key, {"entryexists": true, "data": [1]});
            return true;
        }
    }
    return false;
}

/**
* Initializes and manages WordPress admin menu visibility
*
* @async
* @returns {Promise<void>}
*
* @description
* Controls which menu items are visible based on:
* - User authentication status
* - Registration existence
* - Notification setup
* - Active checkups
*
* Menu items (indices):
* 0: Settings
* 1: Notifications
* 2: Checkups
* 3: Checkup Results
* 4: Dashboard
*
* Visibility rules:
* - Not authenticated: Show only Settings
* - No registration: Show only Settings
* - No notifications: Hide Results/Dashboard
* - No checkups: Hide Results/Dashboard
* - No active checkups: Hide Results/Dashboard
*/
async function check_main_menu() {
    const menu_list = Array.from(getElById('toplevel_page_website_checkups_menu').querySelector('.wp-submenu').querySelectorAll('li'));
    menu_list.shift();

    menu_list.forEach((item, index) => {
        if (!item?.id) {
            item.id = 'menu-item-' + (index + 1);
            setClass(item, '', 'd-none');
        }
    });

    if (!wc_user_data['token']) {
        close_loader_safe();
        setClass(menu_list[0], 'd-none', 'd-block');
    } else {
        const registration_exists = await check_data_exists('registrations');
        if (!registration_exists) {
            setClass(menu_list[0], 'd-none', 'd-block');
        } else {
            const notification_data = await manage_api_data('notifications', 'get');
            if (!notification_data?.data || notification_data.data.length <= 0) {
                setClass(menu_list[0], 'd-none', '');
                setClass(menu_list[1], 'd-none', '');
                setClass(menu_list[2], '', 'd-none');
                setClass(menu_list[3], '', 'd-none');
                setClass(menu_list[4], '', 'd-none');
            } else {
                const checkup_data = await manage_api_data('sitecheckups', 'get');
                let active_checkup = false;

                if (!checkup_data?.data || checkup_data.data.length <= 0) {
                    setClass(menu_list[0], 'd-none', 'd-block');
                    setClass(menu_list[1], 'd-none', 'd-block');
                    setClass(menu_list[2], 'd-none', 'd-block');
                    setClass(menu_list[3], 'd-block', 'd-none');
                    setClass(menu_list[4], 'd-block', 'd-none');
                } else {
                    for (const single_checkup_data of checkup_data.data) {
                        if (single_checkup_data.attributes.active === 1) {
                            active_checkup = true;
                            break;
                        }
                    }
                    setClass(menu_list[0], 'd-none', 'd-block');
                    setClass(menu_list[1], 'd-none', 'd-block');
                    setClass(menu_list[2], 'd-none', 'd-block');

                    if (active_checkup) {
                        setClass(menu_list[3], 'd-none', 'd-block');
                        setClass(menu_list[4], 'd-none', 'd-block');
                    } else {
                        setClass(menu_list[3], 'd-block', 'd-none');
                        setClass(menu_list[4], 'd-block', 'd-none');
                    }
                }
            }
        }
    }
}

// ===================================
// LEGACY DATA FUNCTIONS
// ===================================
/**
* Manages local data operations via WordPress AJAX
*
* @param {Object} post_data - Data to send to server
* @param {string} post_data.post_type - Type of operation ('load_form', 'user_register', etc.)
* @returns {void}
*
* @description
* Legacy function for WordPress AJAX communication.
*
* Special handling for 'load_form':
* - Checks localStorage cache first
* - Fetches from server if not cached
* - Caches field metadata (without field_item DOM references)
*
* Special handling for 'user_register':
* - Shows success toast
* - Reloads page after 2 seconds
*
* @example
* manage_local_data({post_type: 'load_form'});
*/
async function manage_local_data(post_data) {
    // Für load_form: Check localStorage first
    if (post_data?.post_type === 'load_form') {
        const cached = get_webcheckups_storage_data('form_fields_data');
        if (cached) {
            // console.log('✅ Using cached form fields from localStorage');
            process_form_data(cached);
            return;
        }
        // console.log('📡 No cache found, fetching from server...');
    }

    post_data.action = 'manage_admin_data';
    post_data.nonce = wc_nonce;

    return fetch(ajaxurl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
        },
        body: new URLSearchParams(post_data)
    }).then(response => response.json())
        .then(async result => {
            if (result?.success && post_data?.post_type === 'load_form') {
                // Process and cache data
                process_form_data(result.data);
                // Store in localStorage (without field_item - only metadata)
                const cache_data = result.data.map(form => ({
                    form_name: form.form_name,
                    fields: form.fields.map(f => ({
                        field: f.field,
                        type: f.type,
                        advanced: f.advanced,
                        required: f.required,
                        valuesDescription: f.valuesDescription,
                        disableFor: f.disableFor,
                        label: f.label
                    }))
                }));
                set_webcheckups_storage_data('form_fields_data', cache_data);
            }
            if (result?.success && post_data?.post_type === 'user_register' && wc_page_class === 'toplevel_page_website_checkups_menu') {
                setTimeout(function () {
                    location.reload();
                }, 2000);

                if (Array.isArray(result?.data?.msg)) {
                    result.data.msg.forEach((single_msg, index) => {
                        show_toast(result?.success, single_msg, (index + 1));
                    });
                } else {
                    show_toast(result?.success, result?.data?.msg, 1);
                }
            } else {
                enable_all_fieldset();
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });
}

/**
* Processes form field definitions and stores in global variable
*
* @param {Array<Object>} data - Array of form definitions from API/cache
* @param {string} data[].form_name - Form identifier
* @param {Array<Object>} data[].fields - Field definitions
* @returns {void}
*
* @description
* Transforms raw form field data into structured format with DOM references.
*
* Special handling:
* - Automatically creates virtual switch fields for specific dropdowns
*   (country, countryReg, lang, langReg) to enable "Show all" toggles
* - Only adds switch fields if corresponding DOM element exists
*
* Stores result in global `wc_form_data` object organized by form name.
*
* @example
* // Input data structure:
* [{
*   form_name: 'registration',
*   fields: [{field: 'email', type: 'text', required: true, ...}]
* }]
*
* // Output in wc_form_data:
* {
*   'registration': [
*     {field_item: DOMElement, field_name: 'email', ...}
*   ]
* }
*/
function process_form_data(data) {
    const tmp_result = {};

    data.forEach(form => {
        const form_name = form.form_name;
        tmp_result[form_name] = [];

        form.fields.forEach(field => {
            const tmp_data = {
                'field_item': getElById(field.field),
                'field_name': field.field,
                'field_type': field.type,
                'field_advanced': field.advanced,
                'field_required': field.required,
                'values_description': field.valuesDescription,
                'disable_for': field.disableFor,
                'detail': field.required && field.label ? field.label + ' is required' : null
            };

            tmp_result[form_name].push(tmp_data);

            // Virtual switch fields for dropdowns with "Show all" toggle
            const switch_fields = ['country', 'countryReg', 'lang', 'langReg'];
            if (switch_fields.includes(field.field)) {
                const switch_field_name = 'switch_' + field.field;
                const switch_element = getElById(switch_field_name);

                // Only add if element exists
                if (switch_element) {
                    const switch_data = {
                        'field_item': switch_element,
                        'field_name': switch_field_name,
                        'field_type': 'switch',
                        'field_advanced': null,
                        'field_required': false,
                        'values_description': null,
                        'disable_for': null,
                        'detail': null
                    };
                    tmp_result[form_name].push(switch_data);
                }
            }
        });
    });

    wc_form_data = tmp_result;
    // console.log('✅ Form data processed:', Object.keys(wc_form_data));
}
