/* /lib/js/cache-statistics.js */
/*!
 * Timeline Statistics Cache for Website Checkups Plugin v1.2.0
 * https://website-checkups.com
 * Timeline-based caching with compression and smart gap detection
 * Released under the GPLv2 license
 */

// ===================================
// CONSTANTS
// ===================================

const STATS_CACHE_PREFIX = 'webcheckups:stats:';
const STATS_CACHE_MAX_AGE_DAYS = 90; // Auto-cleanup after 90 days
const STATS_CACHE_VERSION = '1.0'; // Increment when cache format changes (1.0 = timeline format)

// Timeline cache retention levels (for quota management)
const TIMELINE_RETENTION_LEVELS = [
    { months: 12, label: 'year' },
    { months: 6, label: '6month' },
    { months: 3, label: '3month' },
    { months: 2, label: '2month' },
    { months: 1, label: 'month' },
    { weeks: 2, label: '2weeks' },
    { weeks: 1, label: 'week' }
];
// ===================================
// CACHE VERSION MANAGEMENT
// ===================================
        
/**
  * Checks and migrates cache version if needed
  *
  * @description
  * Clears all statistics cache if version mismatch detected.
  * This prevents errors when cache format changes between plugin versions.
  */
function check_statistics_cache_version() {
    const versionKey = STATS_CACHE_PREFIX + 'version';
    const storedVersion = localStorage.getItem(versionKey);

    if (storedVersion !== STATS_CACHE_VERSION) {
        // console.log(`🔄 Cache version mismatch (${storedVersion} → ${STATS_CACHE_VERSION}), clearing statistics cache...`);
        clear_all_statistics_cache();
        localStorage.setItem(versionKey, STATS_CACHE_VERSION);
    }
}

// Check version on load
check_statistics_cache_version();

// ===================================
// LZ-STRING COMPRESSION
// ===================================
/**
  * LZ-String compression library (minified)
  * https://pieroxy.net/blog/pages/lz-string/index.html
  * MIT License
  */
var LZString=function(){function o(o,r){if(!t[o]){t[o]={};for(var n=0;n<o.length;n++)t[o][o.charAt(n)]=n}return t[o][r]}var r=String.fromCharCode,n="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",e="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-$",t={},i={compressToBase64:function(o){if(null==o)return"";var r=i._compress(o,6,function(o){return n.charAt(o)});switch(r.length%4){default:case 0:return r;case 1:return r+"===";case 2:return r+"==";case 3:return r+"="}},decompressFromBase64:function(r){return null==r?"":""==r?null:i._decompress(r.length,32,function(e){return o(n,r.charAt(e))})},compressToUTF16:function(o){return null==o?"":i._compress(o,15,function(o){return r(o+32)})+" "},decompressFromUTF16:function(o){return null==o?"":""==o?null:i._decompress(o.length,16384,function(r){return o.charCodeAt(r)-32})},compressToUint8Array:function(o){for(var r=i.compress(o),n=new Uint8Array(2*r.length),e=0,t=r.length;t>e;e++){var s=r.charCodeAt(e);n[2*e]=s>>>8,n[2*e+1]=s%256}return n},decompressFromUint8Array:function(o){if(null===o||void 0===o)return i.decompress(o);for(var n=new Array(o.length/2),e=0,t=n.length;t>e;e++)n[e]=256*o[2*e]+o[2*e+1];var s=[];return n.forEach(function(o){s.push(r(o))}),i.decompress(s.join(""))},compressToEncodedURIComponent:function(o){return null==o?"":i._compress(o,6,function(o){return e.charAt(o)})},decompressFromEncodedURIComponent:function(r){return null==r?"":""==r?null:(r=r.replace(/ /g,"+"),i._decompress(r.length,32,function(n){return o(e,r.charAt(n))}))},compress:function(o){return i._compress(o,16,function(o){return r(o)})},_compress:function(o,r,n){if(null==o)return"";var e,t,i,s={},p={},u="",c="",a="",l=2,f=3,h=2,d=[],m=0,v=0;for(i=0;i<o.length;i+=1)if(u=o.charAt(i),Object.prototype.hasOwnProperty.call(s,u)||(s[u]=f++,p[u]=!0),c=a+u,Object.prototype.hasOwnProperty.call(s,c))a=c;else{if(Object.prototype.hasOwnProperty.call(p,a)){if(a.charCodeAt(0)<256){for(e=0;h>e;e++)m<<=1,v==r-1?(v=0,d.push(n(m)),m=0):v++;for(t=a.charCodeAt(0),e=0;8>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1}else{for(t=1,e=0;h>e;e++)m=m<<1|t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t=0;for(t=a.charCodeAt(0),e=0;16>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1}l--,0==l&&(l=Math.pow(2,h),h++),delete p[a]}else for(t=s[a],e=0;h>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1;l--,0==l&&(l=Math.pow(2,h),h++),s[c]=f++,a=String(u)}if(""!==a){if(Object.prototype.hasOwnProperty.call(p,a)){if(a.charCodeAt(0)<256){for(e=0;h>e;e++)m<<=1,v==r-1?(v=0,d.push(n(m)),m=0):v++;for(t=a.charCodeAt(0),e=0;8>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1}else{for(t=1,e=0;h>e;e++)m=m<<1|t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t=0;for(t=a.charCodeAt(0),e=0;16>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1}l--,0==l&&(l=Math.pow(2,h),h++),delete p[a]}else for(t=s[a],e=0;h>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1;l--,0==l&&(l=Math.pow(2,h),h++)}for(t=2,e=0;h>e;e++)m=m<<1|1&t,v==r-1?(v=0,d.push(n(m)),m=0):v++,t>>=1;for(;;){if(m<<=1,v==r-1){d.push(n(m));break}v++}return d.join("")},decompress:function(o){return null==o?"":""==o?null:i._decompress(o.length,32768,function(r){return o.charCodeAt(r)})},_decompress:function(o,n,e){var t,i,s,p,u,c,a,l,f=[],h=4,d=4,m=3,v="",w=[],A={val:e(0),position:n,index:1};for(i=0;3>i;i+=1)f[i]=i;for(p=0,c=Math.pow(2,2),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;switch(t=p){case 0:for(p=0,c=Math.pow(2,8),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;l=r(p);break;case 1:for(p=0,c=Math.pow(2,16),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;l=r(p);break;case 2:return""}for(f[3]=l,s=l,w.push(l);;){if(A.index>o)return"";for(p=0,c=Math.pow(2,m),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;switch(l=p){case 0:for(p=0,c=Math.pow(2,8),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;f[d++]=r(p),l=d-1,h--;break;case 1:for(p=0,c=Math.pow(2,16),a=1;a!=c;)u=A.val&A.position,A.position>>=1,0==A.position&&(A.position=n,A.val=e(A.index++)),p|=(u>0?1:0)*a,a<<=1;f[d++]=r(p),l=d-1,h--;break;case 2:return w.join("")}if(0==h&&(h=Math.pow(2,m),m++),f[l])v=f[l];else{if(l!==d)return null;v=s+s.charAt(0)}w.push(v),f[d++]=s+v.charAt(0),h--,s=v,0==h&&(h=Math.pow(2,m),m++)}}};return i}();"function"==typeof define&&define.amd?define(function(){return LZString}):"undefined"!=typeof module&&null!=module&&(module.exports=LZString);

/**
 * Compresses statistics data using LZ-String
 *
 * @param {Object} data - Statistics data object
 * @returns {string} Compressed string
 */
function compress_statistics(data) {
    const json = JSON.stringify(data);
    const compressed = LZString.compressToUTF16(json);
    const compressionRatio = ((1 - compressed.length / json.length) * 100).toFixed(1);
    // console.log(`🗜️ Compressed: ${json.length} → ${compressed.length} bytes (${compressionRatio}% saved)`);
    return compressed;
}

/**
 * Decompresses statistics data using LZ-String
 *
 * @param {string} compressed - Compressed string
 * @returns {Object|null} Decompressed data object or null on error
 */
function decompress_statistics(compressed) {
    try {
        const json = LZString.decompressFromUTF16(compressed);
        if (!json) {
            console.error('❌ Decompression returned null');
            return null;
        }

        return JSON.parse(json);
    } catch (e) {
        console.error('❌ Decompression error:', e);
        return null;
    }
}

// ===================================
// DATE UTILITIES
// ===================================

/**
 * Normalizes timestamp to minute precision
 * Removes seconds and milliseconds for consistent comparison
 *
 * @param {Date|string} timestamp - Date object or ISO string
 * @returns {string} Normalized timestamp (format: "2026-01-01T10:00")
 */
function normalize_to_minute(timestamp) {
    const date = typeof timestamp === 'string' ? new Date(timestamp) : timestamp;
    date.setSeconds(0, 0);
    // Return only YYYY-MM-DDTHH:mm (no seconds, no timezone)
    return date.toISOString().substring(0, 16);
}

/**
 * Parses date string to YYYY-MM-DD format
 *
 * @param {string|Date} date - Date to parse
 * @returns {string} YYYY-MM-DD format
 */
function get_date_key(date) {
    const d = new Date(date);
    return d.getFullYear() + '-' +
        String(d.getMonth() + 1).padStart(2, '0') + '-' +
        String(d.getDate()).padStart(2, '0');
}

/**
 * Gets today's date key
 *
 * @returns {string} Today in YYYY-MM-DD format
 */
function get_today_key() {
    return get_date_key(new Date());
}

/**
 * Checks if a date is today
 *
 * @param {string} dateKey - Date key in YYYY-MM-DD format
 * @returns {boolean} True if date is today
 */
function is_today(dateKey) {
    return dateKey === get_today_key();
}

/**
 * Checks if a date is in the past
 *
 * @param {string} dateKey - Date key in YYYY-MM-DD format
 * @returns {boolean} True if date is before today
 */
function is_past_date(dateKey) {
    const today = get_today_key();
    return dateKey < today;
}

/**
 * Generates array of date keys between two dates
 *
 * @param {string} fromDate - Start date (ISO format)
 * @param {string} toDate - End date (ISO format, defaults to today)
 * @returns {Array<string>} Array of date keys (YYYY-MM-DD)
 */
function get_date_range(fromDate, toDate = null) {
    const start = new Date(fromDate);
    const end = toDate ? new Date(toDate) : new Date();
    const dates = [];

    // Set to start of day
    start.setHours(0, 0, 0, 0);
    end.setHours(23, 59, 59, 999);

    const current = new Date(start);

    while (current <= end) {
        dates.push(get_date_key(current));
        current.setDate(current.getDate() + 1);
    }

    return dates;
}

// ===================================
// TIMELINE CACHE - NEW FORMAT
// ===================================

/**
 * Calculates user's timezone offset in minutes
 *
 * @returns {number} Offset in minutes (e.g., 60 for UTC+1)
 */
function get_user_timezone_offset() {
    const now = new Date();
    return -now.getTimezoneOffset(); // Negative because getTimezoneOffset returns opposite sign
}

/**
 * Calculates tillDate with timezone-aware padding
 *
 * @param {string} cacheFrom - Existing cache start date (ISO format)
 * @returns {string} tillDate for API request (without Z suffix)
 *
 * @description
 * Ensures the incomplete day/hour in cache becomes complete in user's timezone.
 *
 * Example:
 * - Cache from: 2025-11-24T12:00:00 (UTC)
 * - User TZ: Europe/Berlin (UTC+1)
 * - User sees: 2025-11-24 13:00 (Berlin time)
 * - Need: Full day 2025-11-24 in Berlin (00:00-23:59)
 * - Hourly data for 23:00 Berlin needs checks from 22:30-23:29 Berlin
 * - Result: tillDate = 2025-11-24T23:29:59 (UTC) + 30min safety = 2025-11-24T23:59:59
 */
function calculate_tilldate_for_gap(cacheFrom) {
    const cacheDate = new Date(cacheFrom);
    const userOffset = get_user_timezone_offset(); // minutes

    // For user's complete day: need until 23:59 in user TZ
    // For hourly data: 23:00 hour needs data until 23:29 (30min into next hour)
    // So we need: 23:59 user time + 30min safety = 00:29 next day user time

    // Calculate end of day in user timezone
    const endOfDayUserTZ = new Date(cacheDate);
    endOfDayUserTZ.setUTCHours(23, 59, 59, 0);

    // Add 30 minutes safety margin for hourly data
    endOfDayUserTZ.setMinutes(endOfDayUserTZ.getMinutes() + 30);

    // Adjust for user timezone offset
    // If user is UTC+1, we need data until 23:59 CET = 22:59 UTC
    // So subtract the offset
    endOfDayUserTZ.setMinutes(endOfDayUserTZ.getMinutes() - userOffset);

    // Format as ISO without Z (API interprets as UTC)
    const year = endOfDayUserTZ.getUTCFullYear();
    const month = String(endOfDayUserTZ.getUTCMonth() + 1).padStart(2, '0');
    const day = String(endOfDayUserTZ.getUTCDate()).padStart(2, '0');
    const hours = String(endOfDayUserTZ.getUTCHours()).padStart(2, '0');
    const minutes = String(endOfDayUserTZ.getUTCMinutes()).padStart(2, '0');
    const seconds = String(endOfDayUserTZ.getUTCSeconds()).padStart(2, '0');

    const tillDate = `${year}-${month}-${day}T${hours}:${minutes}:${seconds}`;

    // console.log(`📅 TillDate calculation: cacheFrom=${cacheFrom}, userOffset=${userOffset}min, tillDate=${tillDate}`);

    return tillDate;
}

/**
 * Analyzes gaps in timeline coverage and prepares API requests
 *
 * @param {Object|null} timeline - Cached timeline data
 * @param {string} requestedFrom - Requested start date (ISO format)
 * @returns {Array<Object>} Array of gaps: [{fromDate, tillDate}, ...]
 *
 * @description
 * Determines which date ranges need to be fetched from API.
 *
 * Scenarios:
 * 1. No cache: Full range from requestedFrom to today
 * 2. Gap before: Old data missing
 * 3. Gap after: Today's data needs update
 * 4. Both gaps: Sandwich scenario
 */
function analyze_timeline_gaps(timeline, requestedFrom) {
    const gaps = [];
    const now = new Date();
    // CRITICAL: Ensure requestedFrom is parsed as UTC, not local time
    // If no 'Z' suffix, add it to force UTC interpretation
    const requestedFromUTC = requestedFrom.endsWith('Z') ? requestedFrom : requestedFrom.replace(' ', 'T') + 'Z';
    const requested = new Date(requestedFromUTC);

    if (!timeline) {
        // No cache: fetch everything from requested date to now
        // console.log('📊 No timeline cache: fetching full range');
        gaps.push({
            fromDate: requestedFrom,
            tillDate: null, // Open-ended to today
            reason: 'no_cache'
        });
        return gaps;
    }

    const cacheFrom = new Date(timeline.from);
    const cacheUntil = new Date(timeline.until);

    // console.log('🔍 Gap analysis:', {
    //     requested: requestedFrom,
    //     cacheFrom: timeline.from,
    //     cacheUntil: timeline.until,
    //     cacheSegments: timeline.segments?.length
    // });

    // Determine check interval from LAST REALTIME segment (latest = most current)
    const realtimeSegments = timeline.segments?.filter(s => s.data?.time);
    const realtimeSegment = realtimeSegments?.[realtimeSegments.length - 1];

    const checkInterval = realtimeSegment?.interval || 60; // minutes, default 60
    // console.log('🔍 Check interval from timeline:', checkInterval, 'minutes');

    // Gap BEFORE cache (older data needed - only if gap > interval)
    const gapBeforeMinutes = (cacheFrom - requested) / 1000 / 60;
     // console.log('🔍 Gap before calculation:', {
     //     gapMinutes: gapBeforeMinutes,
     //     checkInterval: checkInterval,
     //     shouldFetch: gapBeforeMinutes > checkInterval
     // });

    if (gapBeforeMinutes > checkInterval) {
        // Check if this is a calendar request asking for roughly the same timespan
        // If requested date is within 24h of cache, it's the same "last day" view
        const hoursDifference = gapBeforeMinutes / 60;
        const isCalendarSameDay = hoursDifference <= 25 && hoursDifference >= 23;

        if (isCalendarSameDay) {
            // console.log(`📊 Calendar request for same day (${Math.round(hoursDifference)}h difference): reusing cache`);
        } else {
            const tillDate = calculate_tilldate_for_gap(timeline.from);

            // console.log(`📊 Gap before cache (${Math.round(gapBeforeMinutes)}min > ${checkInterval}min interval): ${requestedFrom} → ${timeline.from}`);
            gaps.push({
                fromDate: requestedFrom,
                tillDate: tillDate,
                reason: 'gap_before'
            });
        }
    }
    // } else if (gapBeforeMinutes > 0) {
    //     console.log(`📊 Small gap before cache (${Math.round(gapBeforeMinutes)}min ≤ ${checkInterval}min interval): ignoring`);
    // }

    // Gap AFTER cache (today's data update)
    // Use dynamic expires_at if available, otherwise fall back to interval-based check

    const cacheAge = (now - cacheUntil) / 1000; // seconds
    const currentTimestamp = Math.floor(Date.now() / 1000);

// Check if cache has expired based on expires_at
    let needsRefresh = false;
    if (timeline.expires_at && currentTimestamp >= timeline.expires_at) {
        needsRefresh = true;
        // console.log(`⏱️ Cache expired (expires_at: ${timeline.expires_at}, now: ${currentTimestamp})`);
    } else if (timeline.expires_at) {
        // Have expires_at but not expired yet
        const remainingTtl = timeline.expires_at - currentTimestamp;
        // console.log(`⏱️ Cache still fresh (remaining: ${remainingTtl}s)`);
        needsRefresh = false;
    } else {
        // Fallback: use check interval from timeline
        const fallbackSeconds = checkInterval * 60;
        needsRefresh = cacheAge > fallbackSeconds;
        // console.log(`⏱️ No expires_at, using interval fallback (age: ${Math.round(cacheAge)}s, threshold: ${fallbackSeconds}s)`);
     }
    
    // Check if the request is for the SAME or LATER range as cached (no missing old data)
    // If requested >= cacheFrom, it means we already have all requested historical data
    const requestCoversCache = requested >= cacheFrom;
    
    // console.log('🔍 Gap after calculation:', {
    //     needsRefresh: needsRefresh,
    //     requestCoversCache: requestCoversCache,
    //     hasGapBefore: gapBeforeMinutes > checkInterval
    // });

    // If cache is expired/stale, ALWAYS refresh (gap_after)
    if (needsRefresh) {
        // Calculate fromDate: start a bit before cache.until for seamless merge
        // Use 30min before to ensure hourly data completeness
        const gapFrom = new Date(cacheUntil);
        gapFrom.setMinutes(gapFrom.getMinutes() - 30);

        const fromDate = gapFrom.toISOString().replace(/\.\d{3}Z$/, '').replace('Z', '');

        // console.log(`📊 Gap after cache (refresh needed): ${fromDate} → now`);
        gaps.push({
            fromDate: fromDate,
            tillDate: null, // Open-ended to today
            reason: 'gap_after'
        });
    }
    // else {
    //     console.log(`📊 Cache is fresh: no refresh needed`);
    // }

    // console.log('🔍 Total gaps found:', gaps.length, gaps);
    return gaps;
}

// ===================================
// TIMELINE CACHE - MERGE LOGIC
// ===================================

/**
 * Merges new API data into existing timeline
 *
 * @param {Object|null} existingTimeline - Existing cached timeline
 * @param {Array} apiResponses - Array of API responses with gap info
 * @param {string} requestedFrom - Original requested start date
 * @returns {Object} Merged timeline object
 *
 * @description
 * Smart merge strategy:
 * 1. Hourly data: Merge all hourly objects into ONE
 * 2. Realtime: Extend or create new segments, handle overlaps
 * 3. Update from/until boundaries
 */
function merge_timeline_data(existingTimeline, apiResponses, requestedFrom) {
    // console.log('🔀 Starting timeline merge...');
 
    // Start with existing timeline or create new
    const merged = existingTimeline ? {
        checkupId: existingTimeline.checkupId,
        from: existingTimeline.from,
        until: existingTimeline.until,
        checkupMeta: existingTimeline.checkupMeta || null,
        segments: [...existingTimeline.segments]
    } : {
        checkupId: null,
        from: requestedFrom,
        until: new Date().toISOString(),
        checkupMeta: null,
        segments: []
    };
 
    // Process each API response
    apiResponses.forEach((response, idx) => {
        // console.log(`🔀 Merging gap ${idx + 1}/${apiResponses.length}: ${response.gap.reason}`);

         // Extract and update checkup meta from API response
         if (response.meta?.checkup) {
             merged.checkupMeta = {
                 name: response.meta.checkup.name || merged.checkupMeta?.name || null,
                 url: response.meta.checkup.url || merged.checkupMeta?.url || null
             };
             // console.log(`  📋 Updated checkup meta: ${merged.checkupMeta.name || 'N/A'}`);
         }
        
        response.data.forEach(newSegment => {
            // Set checkupId if not set
            if (!merged.checkupId && newSegment.checkup) {
                merged.checkupId = newSegment.checkup;
            }

            // Handle hourly segment
                if (newSegment.data?.hourly) {
                merge_hourly_segment(merged.segments, newSegment, requestedFrom);
            }
            // Handle realtime segment
            else if (newSegment.data?.time && Array.isArray(newSegment.data.time)) {
                 merge_realtime_segment(merged.segments, newSegment);
            }
            // Handle basic segments (failures, YesNo)
            else {
                merge_basic_segment(merged.segments, newSegment);
            }
        });
    });
 
    // Update timeline boundaries
    update_timeline_boundaries(merged);
 
    // console.log(`✅ Merge complete: ${merged.segments.length} segment(s), ${merged.from} → ${merged.until}`);

    return merged;
 }

/**
 * Merges hourly segment into timeline
 *
 * @param {Array} segments - Existing segments array (modified in place)
 * @param {Object} newSegment - New hourly segment from API
 *
 * @description
 * Strategy:
 * 1. Find existing hourly segment (there should be only ONE)
 * 2. Merge hourly objects (overwrite incomplete hours, keep complete)
 * 3. Update segment boundaries
 * 4. Compute daily from hourly for compatibility
 */
function merge_hourly_segment(segments, newSegment, requestedFrom = null) {
    // Find existing hourly segment
    const hourlyIndex = segments.findIndex(s => s.data?.hourly);
    let hourlySegment = hourlyIndex >= 0 ? segments[hourlyIndex] : null;

    if (!hourlySegment) {
        // No existing hourly segment, add new one
        const cleanSegment = {
            from: newSegment.from,
            until: newSegment.until,
            status: newSegment.status,
            counter: newSegment.counter,
            interval: newSegment.interval,
            data: {
                hourly: {...newSegment.data.hourly}
            }
        };

        segments.push(cleanSegment);
        // console.log('  ✅ Created new hourly segment');
        return;
    }

    // Get existing hourly data
    const existingHourly = hourlySegment.data.hourly;

    // Merge hourly data
    const newHourly = newSegment.data.hourly;

    // Determine first and last hour in API response
    const newHourKeys = Object.keys(newHourly).sort();
    const firstNewHour = newHourKeys[0];
    const lastNewHour = newHourKeys[newHourKeys.length - 1];

    let overwrittenCount = 0;
    let addedCount = 0;
    let skippedCount = 0;

    Object.keys(newHourly).forEach(hourKey => {
        const existingHour = existingHourly[hourKey];
        const newHour = newHourly[hourKey];

        // Determine if hour is "complete" in API response
        // First and last hour of API response are potentially incomplete
        const isIncompleteInApi = (hourKey === firstNewHour) || (hourKey === lastNewHour);

        if (!existingHour) {
            // New hour, add it
            existingHourly[hourKey] = {...newHour};
            addedCount++;
        } else if (existingHour.complete === false || isIncompleteInApi === false) {
            // Existing hour is incomplete OR new hour is complete → overwrite
            const wasIncomplete = existingHour.complete === false;
            existingHourly[hourKey] = {...newHour};

            // Remove complete flag if new data is complete
                if (!isIncompleteInApi && existingHourly[hourKey].complete === false) {
                delete existingHourly[hourKey].complete;
            }

            overwrittenCount++;
            if (wasIncomplete) {
                // console.log(`  📝 Updated incomplete hour: ${hourKey}`);
            }
        } else {
            // Existing hour is complete, skip
            skippedCount++;
        }
    });

    // console.log(`  ✅ Hourly merge: ${addedCount} added, ${overwrittenCount} updated, ${skippedCount} skipped`);

    // Update segment boundaries
    const allHourKeys = Object.keys(existingHourly).sort();
    if (allHourKeys.length > 0) {
         // Reconstruct from/until from hour keys (format: "2025-12-26 22:00")
         // Need to parse and add seconds separately
         const firstHourKey = allHourKeys[0]; // "2025-12-26 22:00"
         const lastHourKey = allHourKeys[allHourKeys.length - 1]; // "2025-12-27 23:00"

         // Parse date and hour, then reconstruct with proper format
         hourlySegment.from = firstHourKey.replace(' ', 'T') + ':00Z'; // 22:00 → 22:00:00
         hourlySegment.until = lastHourKey.replace(' ', 'T') + ':59Z'; // 23:00 → 23:59:59

        hourlySegment.counter = allHourKeys.length;

        // Make sure the array reference is updated
        segments[hourlyIndex] = hourlySegment;
    }
}

/**
 * Merges realtime segment into timeline
 *
 * @param {Array} segments - Existing segments array (modified in place)
 * @param {Object} newSegment - New realtime segment from API
 *
 * @description
 * Strategy:
 * 1. Find overlapping segments (same interval, overlapping time range)
 * 2. If overlap exists: extend or merge
 * 3. If no overlap: add as new segment
 */
function merge_realtime_segment(segments, newSegment) {
    const newFrom = new Date(newSegment.from);
    const newUntil = new Date(newSegment.until);
    const newInterval = newSegment.interval;

    // console.log(`🔀 Merging realtime: ${newSegment.from} → ${newSegment.until}, ${newSegment.counter} checks`);

    // Find potentially overlapping segments (same status, same interval)
    const overlapping = segments.filter(s =>
        s.data?.time &&
        s.status === newSegment.status &&
        s.interval === newInterval
    );

    if (overlapping.length === 0) {
        // No overlap, add as new segment
        segments.push({...newSegment});
        // console.log('  ✅ Added new realtime segment');
        return;
    }

    // Check for actual time overlap
    let merged = false;

    for (const existing of overlapping) {
        const existFrom = new Date(existing.from);
        const existUntil = new Date(existing.until);

        // Check if segments are adjacent or overlapping
        const gap = (newFrom - existUntil) / 1000 / 60; // minutes
        const isAdjacent = gap >= 0 && gap <= newInterval * 1.5; // Allow small gap
        const isOverlapping = newFrom <= existUntil && newUntil >= existFrom;

        if (isAdjacent || isOverlapping) {
            // console.log(`  🔗 Found adjacent/overlapping segment, extending...`);

            // Extend existing segment
            if (isOverlapping) {
                // Overlap: Use timestamp-based deduplication (THE ONLY SAFE METHOD)
                const overlapStart = newFrom > existFrom ? newFrom : existFrom;
                const overlapEnd = newUntil < existUntil ? newUntil : existUntil;

                console.log(`  🔍 Overlap detected:`, {
                    overlapStart: overlapStart.toISOString(),
                    overlapEnd: overlapEnd.toISOString(),
                    existingChecks: existing.data.time.length,
                    newChecks: newSegment.data.time.length
                });

                // Step 1: Build set of existing timestamps
                const existingTimestamps = new Set();
                let timeObj = new Date(existing.from);
                const existIntervals = existing.data.intervals ?? {};

                for (let i = 0; i < existing.data.time.length; i++) {
                    const timestamp = normalize_to_minute(timeObj);
                    existingTimestamps.add(timestamp);

                    const interval = existIntervals[i.toString()] ?? existing.interval;
                    timeObj.setMinutes(timeObj.getMinutes() + interval);
                }

                // Step 2: Check which new checks are NOT in existing
                const checksToAdd = [];
                const newIntervalsToAdd = {};
                const newErrRefsToAdd = {};

                timeObj = new Date(newSegment.from);
                const newIntervals = newSegment.data.intervals ?? {};
                const newErrRefs = newSegment.data.errReferences ?? {};
                let checksInOverlap = 0;


                for (let i = 0; i < newSegment.data.time.length; i++) {
                    const timestamp = normalize_to_minute(timeObj);

                    if (existingTimestamps.has(timestamp)) {
                        // This check already exists - skip it
                        checksInOverlap++;
                    } else {
                        // New check - add it
                        checksToAdd.push(newSegment.data.time[i]);

                        // Store interval if exists
                        if (newIntervals[i.toString()] !== undefined) {
                            newIntervalsToAdd[checksToAdd.length - 1] = newIntervals[i.toString()];
                        } else {
                            // Store default explicitly to avoid using wrong existing.interval
                            newIntervalsToAdd[checksToAdd.length - 1] = newSegment.interval;
                        }

                        // Copy error reference if exists
                        if (newErrRefs[i.toString()] !== undefined) {
                            newErrRefsToAdd[checksToAdd.length - 1] = newErrRefs[i.toString()];
                        }
                    }

                    const interval = newIntervals[i.toString()] ?? newSegment.interval;
                    timeObj.setMinutes(timeObj.getMinutes() + interval);
                }

                console.log(`  📊 Timestamp-based deduplication:`, {
                    checksInOverlap: checksInOverlap,
                    checksToAdd: checksToAdd.length,
                    totalNew: newSegment.data.time.length
                });

                if (checksToAdd.length > 0) {
                    // Split new checks into BEFORE and AFTER groups
                    const checksToAddBefore = [];
                    const checksToAddAfter = [];

                    const existingFrom = new Date(existing.from);
                    const existingUntil = new Date(existing.until);

                    // Rebuild timeObj to assign each check to correct group
                    timeObj = new Date(newSegment.from);
                    let checksToAddIndex = 0;

                    for (let i = 0; i < newSegment.data.time.length; i++) {
                        const timestamp = normalize_to_minute(timeObj);

                        if (!existingTimestamps.has(timestamp)) {
                            // This is a new check - determine BEFORE or AFTER
                            const checkData = {
                                value: checksToAdd[checksToAddIndex],
                                timeObj: new Date(timeObj),
                                interval: newIntervalsToAdd[checksToAddIndex],
                                errRef: newErrRefsToAdd[checksToAddIndex]
                            };

                            if (timeObj < existingFrom) {
                                checksToAddBefore.push(checkData);
                            } else if (timeObj > existingUntil) {
                                checksToAddAfter.push(checkData);
                            } else {
                                // Should not happen (API delivers gapless data)
                                    console.warn(`  ⚠️ Check in middle: ${timestamp}`);
                                checksToAddAfter.push(checkData);
                            }

                            checksToAddIndex++;
                        }

                        const interval = newIntervals[i.toString()] ?? newSegment.interval;
                        timeObj.setMinutes(timeObj.getMinutes() + interval);
                    }

                    // Sort both groups ASC (chronological)
                    checksToAddBefore.sort((a, b) => a.timeObj - b.timeObj);
                    checksToAddAfter.sort((a, b) => a.timeObj - b.timeObj);

                    console.log(`  📊 Merge groups:`, {
                        checksToAddBefore: checksToAddBefore.length,
                        existingChecks: existing.data.time.length,
                        checksToAddAfter: checksToAddAfter.length
                    });

                    // Build new arrays: [BEFORE, EXISTING, AFTER]
                    const newTimeArray = [];
                    const rebuiltIntervals = {};  // ← KEIN Namenskonflikt!
                    const rebuiltErrRefs = {};
                    let idx = 0;

                    // Add BEFORE checks
                    for (const check of checksToAddBefore) {
                        newTimeArray.push(check.value);
                        // Store interval (sparse - only if not undefined)
                        if (check.interval !== undefined) {
                            rebuiltIntervals[idx.toString()] = check.interval;
                        }
                        if (check.errRef !== undefined) {
                            rebuiltErrRefs[idx.toString()] = check.errRef;
                        }
                        idx++;
                    }

                    // Add EXISTING checks (preserve sparse intervals)
                    const existIntervals = existing.data.intervals ?? {};
                    const existErrRefs = existing.data.errReferences ?? {};
                    for (let i = 0; i < existing.data.time.length; i++) {
                        newTimeArray.push(existing.data.time[i]);
                        // Copy interval only if exists (keep sparse)
                        if (existIntervals[i.toString()] !== undefined) {
                            rebuiltIntervals[idx.toString()] = existIntervals[i.toString()];
                        }
                        if (existErrRefs[i.toString()] !== undefined) {
                            rebuiltErrRefs[idx.toString()] = existErrRefs[i.toString()];
                        }
                        idx++;
                    }

                    // Add AFTER checks
                    for (const check of checksToAddAfter) {
                        newTimeArray.push(check.value);
                        if (check.interval !== undefined) {
                            rebuiltIntervals[idx.toString()] = check.interval;
                        }
                        if (check.errRef !== undefined) {
                            rebuiltErrRefs[idx.toString()] = check.errRef;
                        }
                        idx++;
                    }

                    // Replace existing data
                    existing.data.time = newTimeArray;
                    existing.data.intervals = rebuiltIntervals;
                    if (Object.keys(rebuiltErrRefs).length > 0) {
                        existing.data.errReferences = rebuiltErrRefs;
                    }
                    existing.counter = newTimeArray.length;

                    // Update from/until based on actual data
                    if (checksToAddBefore.length > 0) {
                        existing.from = checksToAddBefore[0].timeObj.toISOString().replace(/\.\d{3}Z$/, '.000000Z');
                    }
                    if (checksToAddAfter.length > 0) {
                        existing.until = checksToAddAfter[checksToAddAfter.length - 1].timeObj.toISOString().replace(/\.\d{3}Z$/, '.000000Z');
                    }

                    // VALIDATION: Check if merge is correct
                    let calculatedTimeObj = new Date(existing.from);
                    const mergedIntervals = existing.data.intervals ?? {};

                    for (let i = 0; i < existing.data.time.length - 1; i++) {
                        const interval = mergedIntervals[i.toString()] ?? existing.interval;
                        calculatedTimeObj.setMinutes(calculatedTimeObj.getMinutes() + interval);
                    }
                    // Normalize to minute precision (cut seconds like in timestamp comparison)
                    const calculatedUntil = normalize_to_minute(calculatedTimeObj) + ':00.000000Z';
                    const apiUntil = existing.until;

                    // Normalize both to minute precision
                    const calcNorm = calculatedUntil;
                    const apiNorm = normalize_to_minute(apiUntil) + ':00.000000Z';

                    if (calcNorm !== apiNorm) {
                        console.error('⚠️⚠️⚠️ MERGE ERROR DETECTED ⚠️⚠️⚠️');
                        console.error('Our calculated until differs from API until!');
                        console.error({
                            ourCalculation: calcNorm,
                            apiValue: apiNorm,
                            difference_minutes: Math.round((new Date(apiNorm) - new Date(calcNorm)) / 1000 / 60),
                            segmentFrom: existing.from,
                            totalChecks: existing.counter
                        });
                    } else {
                        console.log(`  ✅ Validation passed: calculated until matches API until`);
                    }

                    console.log(`  ✅ Merged: ${checksToAddBefore.length} before + ${existing.data.time.length - checksToAddBefore.length - checksToAddAfter.length} existing + ${checksToAddAfter.length} after`);
                    console.log(`  📊 Result: ${existing.counter} total checks, from=${existing.from}, until=${existing.until}`);
                } else {
                    console.log(`  ⭐ All checks already in cache, skipped`);
                }
            } else {
                // Adjacent: Add all new data
                existing.data.time.push(...newSegment.data.time);
                existing.counter = existing.data.time.length;

                if (newSegment.data.intervals) {
                    existing.data.intervals = existing.data.intervals || {};
                    const offset = existing.data.time.length - newSegment.data.time.length;
                    Object.keys(newSegment.data.intervals).forEach(key => {
                        const newKey = (parseInt(key) + offset).toString();
                        existing.data.intervals[newKey] = newSegment.data.intervals[key];
                    });
                }
                if (newSegment.data.errReferences) {
                    existing.data.errReferences = existing.data.errReferences || {};
                    const offset = existing.data.time.length - newSegment.data.time.length;
                    Object.keys(newSegment.data.errReferences).forEach(key => {
                        const newKey = (parseInt(key) + offset).toString();
                        existing.data.errReferences[newKey] = newSegment.data.errReferences[key];
                    });
                }

                // VALIDATION für Adjacent auch
                let calculatedTimeObj = new Date(existing.from);
                const mergedIntervals = existing.data.intervals ?? {};

                for (let i = 0; i < existing.data.time.length - 1; i++) {
                    const interval = mergedIntervals[i.toString()] ?? existing.interval;
                    calculatedTimeObj.setMinutes(calculatedTimeObj.getMinutes() + interval);
                }

                const calculatedUntil = normalize_to_minute(calculatedTimeObj) + ':00.000000Z';
                const apiUntil = newSegment.until;
                const calcNorm = calculatedUntil;
                const apiNorm = normalize_to_minute(apiUntil) + ':00.000000Z';

                if (calcNorm !== apiNorm) {
                    console.error('⚠️⚠️⚠️ MERGE ERROR DETECTED (Adjacent) ⚠️⚠️⚠️');
                    console.error({
                        ourCalculation: calcNorm,
                        apiValue: apiNorm,
                        difference_minutes: Math.round((new Date(apiNorm) - new Date(calcNorm)) / 1000 / 60)
                    });
                }

                existing.until = newSegment.until;
                console.log(`  ✅ Extended with ${newSegment.data.time.length} checks`);
            }

            merged = true;
            break;
        }
    }

    // VALIDATION: Check for future data (even 1 minute is a bug!)
    const now = new Date();
    for (const segment of segments) {
        if (segment.data?.time) {
            const segmentUntil = new Date(segment.until);
            if (segmentUntil > now) {
                const futureMinutes = Math.round((segmentUntil - now) / 1000 / 60);
                if (futureMinutes > 0) {
                    console.error('⚠️⚠️⚠️ FUTURE DATA DETECTED ⚠️⚠️⚠️');
                    console.error('Segment contains data in the future!');
                    console.error({
                        segmentUntil: segment.until,
                        now: now.toISOString(),
                        futureMinutes: futureMinutes,
                        segmentFrom: segment.from,
                        checks: segment.counter
                    });
                    console.error('This indicates a bug in our merge logic or interval calculation!');
                }
            }
        }
    }

    if (!merged) {
        // No adjacent segment found, add as new
        segments.push({...newSegment});
        console.log('  ✅ Added as separate segment');
    }
}

/**
 * Merges basic segment (failure, YesNo) into timeline
 *
 * @param {Array} segments - Existing segments array (modified in place)
 * @param {Object} newSegment - New basic segment from API
 */
function merge_basic_segment(segments, newSegment) {
    console.log(`🔀 Merging basic segment: ${newSegment.from}, status=${newSegment.status}`);

    // Check if identical segment already exists
    const exists = segments.some(s =>
        s.from === newSegment.from &&
        s.until === newSegment.until &&
        s.status === newSegment.status
    );

    if (!exists) {
        segments.push({...newSegment});
        console.log('  ✅ Added basic segment');
    } else {
        console.log('  ⏭️ Segment already exists, skipped');
    }
}

/**
 * Updates timeline from/until boundaries based on segments
 *
 * @param {Object} timeline - Timeline object (modified in place)
 */
function update_timeline_boundaries(timeline) {
    if (timeline.segments.length === 0) {
        return;
    }

    let earliestFrom = null;
    let latestUntil = null;

    timeline.segments.forEach(segment => {
        const from = new Date(segment.from);
        const until = new Date(segment.until);

        if (!earliestFrom || from < earliestFrom) {
            earliestFrom = from;
        }
        if (!latestUntil || until > latestUntil) {
            latestUntil = until;
        }
    });

    if (earliestFrom) {
        timeline.from = earliestFrom.toISOString();
    }
    if (latestUntil) {
        timeline.until = latestUntil.toISOString();
    }

    console.log(`📊 Timeline boundaries updated: ${timeline.from} → ${timeline.until}`);
}

/**
 * Main timeline cache loader with smart API calls
 *
 * @async
 * @param {number} checkupId - Checkup ID
 * @param {string} fromDate - Start date (ISO format, e.g., "2025-11-24T00:00:00")
 * @param {Function} apiFetch - API fetch function (checkupId, fromDate, tillDate) => Promise
 * @param {string} variantNeeded - Optional: 'daily', 'hourly', or 'realtime' for filtering
 * @returns {Promise<Object>} API response format with data and meta
 *
 * @description
 * Smart loading strategy:
 * 1. Load timeline cache
 * 2. Analyze coverage gaps
 * 3. Fetch missing data from API (with smart tillDate)
 * 4. Merge new data with cache
 * 5. Return combined result
 */
async function load_statistics_with_timeline_cache(checkupId, fromDate, apiFetch, variantNeeded = null) {
    console.log(`🔍 Timeline load: checkup=${checkupId}, from=${fromDate}`);

    // Step 1: Load existing timeline cache
    const timeline = load_timeline_cache(checkupId);

    // Step 2: Analyze what we need to fetch
    const gaps = analyze_timeline_gaps(timeline, fromDate);

    if (gaps.length === 0 && timeline) {
        // Perfect cache hit!
        console.log('🎯 Perfect timeline cache hit!');

        // Filter segments by requested date range
        const filteredSegments = filter_timeline_segments(timeline.segments, fromDate, variantNeeded);

         // Close loader for cache hit
         if (typeof close_loader_safe === 'function') {
             close_loader_safe();
         }

        return {
            data: filteredSegments,
            meta: {
                ttl: 3600,
                cached: true,
                cache_hit_ratio: 1.0,
                timeline_from: timeline.from,
                timeline_until: timeline.until,
                checkup: timeline.checkupMeta || {}
            }
        };
    }

    // Step 3: Fetch missing data from API
    console.log(`📡 Fetching ${gaps.length} gap(s) from API`);

    const apiResponses = [];
    let lastTtl = null;

    for (const gap of gaps) {
        const logMsg = gap.tillDate
            ? `📡 API call: ${gap.fromDate} → ${gap.tillDate} (${gap.reason})`
            : `📡 API call: ${gap.fromDate} → now (${gap.reason})`;
        console.log(logMsg);

        try {
            const response = await apiFetch(checkupId, gap.fromDate, gap.tillDate);

            if (response?.data) {
                // Extract TTL from API meta
                const ttl = response?.meta?.stat?.cache || 3600;
                lastTtl = ttl;

                apiResponses.push({
                    data: response.data,
                    meta: response.meta,
                    ttl: ttl,
                    gap: gap
                });

                console.log(`✅ API response: ${response.data.length} segment(s), TTL=${ttl}s`);
            }
        } catch (error) {
            console.error(`❌ API fetch failed for gap ${gap.fromDate}:`, error);
        }
    }

    if (apiResponses.length === 0) {
        console.warn('⚠️ No API responses, returning cached data if available');

        if (timeline) {
            const filteredSegments = filter_timeline_segments(timeline.segments, fromDate);

            // Close loader even on error
            if (typeof close_loader_safe === 'function') {
                close_loader_safe();
            }

            return {
                data: filteredSegments,
                meta: {
                    ttl: 3600,
                    cached: true,
                    error: true
                }
            };
        }

        // Close loader before returning error
        if (typeof close_loader_safe === 'function') {
            close_loader_safe();
        }

        return {
            data: [],
            meta: {
                error: true,
                message: 'No cache and API fetch failed'
            }
        };
    }

    // Step 4: Merge new data with existing timeline
    const mergedTimeline = merge_timeline_data(timeline, apiResponses, fromDate);

    // Step 5: Save merged timeline with expires_at from latest API response
    const expiresAt = lastTtl ? Math.floor(Date.now() / 1000) + lastTtl : null;
    save_timeline_cache(checkupId, mergedTimeline, expiresAt);

    // Step 6: Filter and return
    const filteredSegments = filter_timeline_segments(mergedTimeline.segments, fromDate, variantNeeded);

    // Close loader after successful merge
    if (typeof close_loader_safe === 'function') {
        close_loader_safe();
    }

    return {
        data: filteredSegments,
        meta: {
            ttl: lastTtl || 3600,
            cached: false,
            gaps_fetched: gaps.length,
            timeline_from: mergedTimeline.from,
            timeline_until: mergedTimeline.until,
            total_segments: mergedTimeline.segments.length,
            returned_segments: filteredSegments.length,
            checkup: mergedTimeline.checkupMeta || {}
        }
    };
}

/**
 * Filters realtime segments based on quality criteria
 *
 * @param {Array} segments - Segments to filter (already time-range filtered)
 * @param {Date} requestedFrom - Start of requested time range
 * @param {Date} now - Current time
 * @returns {Array} Filtered segments (realtime removed if quality insufficient)
 *
 * @description
 * Quality criteria for realtime segments:
 * 1. Size: Remove segments with >= 5000 checks
 * 2. Coverage: Compare combined realtime coverage with hourly coverage
 *    - If no hourly: require 90% absolute coverage
 *    - If hourly exists: require 90% of hourly coverage
 *    - Decision is all-or-nothing for ALL realtime segments
 */
function filter_realtime_quality(segments, requestedFrom, now) {
    // Find hourly segment to calculate its coverage
    const hourlySegment = segments.find(s => s.data?.hourly);
    let hourlyCoverage = null;

    if (hourlySegment) {
        const hourlyFrom = new Date(hourlySegment.from);
        const hourlyUntil = new Date(hourlySegment.until);

        const hourlyStart = requestedFrom > hourlyFrom ? requestedFrom : hourlyFrom;
        const hourlyEnd = now < hourlyUntil ? now : hourlyUntil;

        const totalRequestedDuration = now - requestedFrom;
        const hourlyCoverageDuration = hourlyEnd - hourlyStart;
        hourlyCoverage = hourlyCoverageDuration / totalRequestedDuration;

        console.log(`  📊 Hourly coverage: ${Math.round(hourlyCoverage * 100)}% (${Math.round(hourlyCoverageDuration / 1000 / 60 / 60)}h / ${Math.round(totalRequestedDuration / 1000 / 60 / 60)}h)`);
    }

    // Collect all realtime segments
    const realtimeSegments = segments.filter(s => s.data?.time);

    if (realtimeSegments.length === 0) {
        // No realtime segments to filter
        return segments;
    }

    // ===================================
    // STEP 1: SIZE CHECK (per segment)
    // ===================================
    const oversizedSegments = new Set();

    realtimeSegments.forEach(segment => {
        const checkCount = segment.data.time.length;
        if (checkCount >= 5000) {
            oversizedSegments.add(segment);
            console.log(`  🚫 Realtime size check: ${checkCount} checks >= 5000 (${segment.from})`);
        }
    });

    // ===================================
    // STEP 2: COVERAGE CHECK (combined across all segments)
    // ===================================
    let shouldRemoveAllRealtime = false;

    if (realtimeSegments.length > 0) {
        // Calculate combined coverage using Set to deduplicate overlaps
        const coveredMinutes = new Set();

        realtimeSegments.forEach(segment => {
            const segmentFrom = new Date(segment.from);
            const segmentUntil = new Date(segment.until);

            // Only count the part within requested range
            const effectiveStart = segmentFrom > requestedFrom ? segmentFrom : requestedFrom;
            const effectiveEnd = segmentUntil < now ? segmentUntil : now;

            // Add each minute to Set (auto-deduplicates)
            let current = new Date(effectiveStart);
            while (current < effectiveEnd) {
                const minuteKey = current.toISOString().substring(0, 16);
                coveredMinutes.add(minuteKey);
                current.setMinutes(current.getMinutes() + 1);
            }
        });

        // Calculate combined coverage percentage
        const totalRequestedMinutes = (now - requestedFrom) / 1000 / 60;
        const coveredMinutesCount = coveredMinutes.size;
        const combinedRealtimeCoverage = coveredMinutesCount / totalRequestedMinutes;

        console.log(`  📊 Combined realtime coverage: ${Math.round(combinedRealtimeCoverage * 100)}% (${Math.round(coveredMinutesCount / 60)}h / ${Math.round(totalRequestedMinutes / 60)}h) across ${realtimeSegments.length} segment(s)`);

        // Determine threshold
        let coverageThreshold;
        let thresholdReason;

        if (hourlyCoverage !== null) {
            coverageThreshold = hourlyCoverage * 0.90;
            thresholdReason = `90% of hourly (${Math.round(hourlyCoverage * 100)}%)`;
        } else {
            coverageThreshold = 0.90;
            thresholdReason = '90% absolute';
        }

        // All-or-nothing decision
        if (combinedRealtimeCoverage < coverageThreshold) {
            shouldRemoveAllRealtime = true;
            const thresholdPercent = Math.round(coverageThreshold * 100);
            const realtimePercent = Math.round(combinedRealtimeCoverage * 100);
            console.log(`  🚫 Combined realtime coverage ${realtimePercent}% < ${thresholdPercent}% ${thresholdReason} → removing ALL ${realtimeSegments.length} realtime segment(s)`);
        } else {
            console.log(`  ✅ Combined realtime coverage ${Math.round(combinedRealtimeCoverage * 100)}% >= ${Math.round(coverageThreshold * 100)}% → keeping realtime`);
        }
    }

    // ===================================
    // STEP 3: APPLY FILTER
    // ===================================
    let removedBySize = 0;
    let removedByCoverage = 0;

    const filtered = segments.filter(segment => {
        // Keep non-realtime segments
        if (!segment.data?.time) {
            return true;
        }

        // Remove oversized segments
        if (oversizedSegments.has(segment)) {
            removedBySize++;
            return false;
        }

        // Remove all realtime if coverage insufficient
        if (shouldRemoveAllRealtime) {
            removedByCoverage++;
            return false;
        }

        return true;
    });

    const removedTotal = segments.length - filtered.length;
    if (removedTotal > 0) {
        console.log(`  🎯 Realtime quality filter: removed ${removedTotal} segments (${removedBySize} by size, ${removedByCoverage} by coverage)`);
    }

    return filtered;
}

/**
 * Filters timeline segments by date range
 *
 * @param {Array} segments - Timeline segments
 * @param {string} fromDate - Start date filter (ISO format)
 * @param {string} variantNeeded - Optional: 'daily', 'hourly', or 'realtime' to filter segment types
 * @returns {Array} Filtered segments
 *
 * @description
 * Returns only segments that overlap with requested date range.
 * If variantNeeded is specified, also filters by data availability:
 * - 'daily': Returns only segments with daily or hourly data (no realtime-only)
 * - 'hourly': Returns only segments with hourly data (no realtime-only, no daily-only)
 * - 'realtime': Returns all segments (no filtering by type)
 * Used when serving from cache.
 */
function filter_timeline_segments(segments, fromDate, variantNeeded = null) {
    if (!segments || segments.length === 0) {
        return [];
    }

    const requestedFrom = new Date(fromDate);
    const now = new Date();

    console.log('🔍 Filter input:', {
        fromDate: fromDate,
        requestedFrom: requestedFrom.toISOString(),
        now: now.toISOString(),
        totalSegments: segments.length
    });

    // Debug segments
    segments.forEach((segment, idx) => {
        const segmentFrom = new Date(segment.from);
        const segmentUntil = new Date(segment.until);
        let checkCount = 0;
        if (segment.data?.time) {
            checkCount = segment.data.time.length;
        } else if (segment.data?.hourly) {
            checkCount = Object.values(segment.data.hourly).reduce((sum, h) => sum + (h.cnt || 0), 0);
        }

        console.log(`🔍 PRE-FILTER Segment ${idx}:`, {
            hasHourly: !!segment.data?.hourly,
            hasDaily: !!segment.data?.daily,
            hasTime: !!segment.data?.time,
            checkCount: checkCount,
            from: segment.from,
            until: segment.until
        });
    });

    // ===================================
    // STEP 1: TIME RANGE FILTER
    // ===================================
    const timeFiltered = segments.filter(segment => {
        const segmentFrom = new Date(segment.from);
        const segmentUntil = new Date(segment.until);
        return segmentUntil >= requestedFrom && segmentFrom <= now;
    });

    console.log(`  ✅ Time range filter: ${timeFiltered.length}/${segments.length} segments`);

    // ===================================
    // STEP 2: REALTIME QUALITY FILTER
    // ===================================
    const qualityFiltered = filter_realtime_quality(timeFiltered, requestedFrom, now);

    console.log(`🔍 After quality filter: ${qualityFiltered.length}/${segments.length} segments`);

    // ===================================
    // STEP 3: VARIANT FILTER (OPTIONAL)
    // ===================================
    let variantFiltered = qualityFiltered;

    if (variantNeeded === 'daily' || variantNeeded === 'hourly') {
        const beforeFilter = qualityFiltered.length;

        variantFiltered = qualityFiltered.filter(segment => {
            const hasDaily = !!segment.data?.daily;
            const hasHourly = !!segment.data?.hourly;

            if (variantNeeded === 'daily') {
                return hasDaily || hasHourly;
            } else if (variantNeeded === 'hourly') {
                return hasHourly;
            }

            return true;
        });

        const removed = beforeFilter - variantFiltered.length;
        if (removed > 0) {
            console.log(`  🔍 Variant filter (${variantNeeded}): removed ${removed} realtime-only segments`);
        }
    }

    // ===================================
    // STEP 4: DAILY COMPUTATION
    // ===================================
    const result = variantFiltered.map(segment => {
        if (segment.data?.hourly && typeof compute_daily_from_hourly_user_tz === 'function') {
            // Filter hourly data by fromDate
            const requestedFromTimestamp = new Date(fromDate.replace(' ', 'T') + 'Z').getTime();
            const filteredHourly = {};

            Object.keys(segment.data.hourly).forEach(hourKey => {
                const hourTimestamp = new Date(hourKey.replace(' ', 'T') + ':00Z').getTime();
                if (hourTimestamp >= requestedFromTimestamp) {
                    filteredHourly[hourKey] = segment.data.hourly[hourKey];
                }
            });

            const sortedFilteredHourly = {};
            Object.keys(filteredHourly).sort().forEach(hourKey => {
                sortedFilteredHourly[hourKey] = filteredHourly[hourKey];
            });

            const segmentCopy = {
                ...segment,
                data: {
                    ...segment.data,
                    hourly: sortedFilteredHourly,
                    daily: compute_daily_from_hourly_user_tz(sortedFilteredHourly, fromDate)
                }
            };

            console.log(`  📊 Computed daily: ${Object.keys(segmentCopy.data.daily).length} days from ${Object.keys(filteredHourly).length} hours`);
            return segmentCopy;
        }

        // Sort time array if exists
        if (segment.data?.time && Array.isArray(segment.data.time)) {
            return {
                ...segment,
                data: {
                    ...segment.data,
                    time: [...segment.data.time].sort((a, b) => {
                        const dateA = new Date(a.performed || a.time);
                        const dateB = new Date(b.performed || b.time);
                        return dateA - dateB;
                    })
                }
            };
        }

        return segment;
    });

    // Debug final result
    result.forEach((segment, idx) => {
        console.log(`🔍 Segment ${idx}:`, {
            hasHourly: !!segment.data?.hourly,
            hasDaily: !!segment.data?.daily,
            hasTime: !!segment.data?.time,
            dailyDates: segment.data?.daily ? Object.keys(segment.data.daily) : [],
            from: segment.from,
            until: segment.until
        });

        if (segment.data?.daily) {
            console.log(`  📊 Daily data for segment ${idx}:`, segment.data.daily);
        }
    });

    console.log(`✅ Final result: ${result.length}/${segments.length} segments for range ${fromDate} → now`);

    return result;
}

/**
 * Saves statistics timeline to cache with compression
 *
 * @param {number} checkupId - Checkup ID
 * @param {Object} timelineData - Timeline data structure
 * @param {number|null} [expiresAt=null] - Optional expiry timestamp from API (current_time + TTL)
 * @returns {boolean} Success status
 *
 * @description
 * Timeline structure:
 * {
 *   checkupId: 33,
 *   from: "2025-11-24T12:00:00",  // Oldest data point (UTC)
 *   until: "2025-12-23T15:40:15", // Newest data point (UTC)
 *   expires_at: timestamp,        // Expiry timestamp (from API meta.stat.cache)
 *   segments: [
 *     {from, until, interval: 60, data: {hourly: {...}}},  // ONE hourly object for ALL days
 *     {from, until, status: 1, data: {time: [...]}},       // Realtime segments
 *     {from, until, status: 0, description: "..."}         // Failure segments
 *   ]
 * }
 */
function save_timeline_cache(checkupId, timelineData, expiresAt = null) {
    const cacheKey = `${STATS_CACHE_PREFIX}${checkupId}:timeline`;

    try {
         // Add expires_at if provided
         const dataToCache = {...timelineData};
         if (expiresAt !== null && expiresAt > 0) {
             dataToCache.expires_at = expiresAt;
         }

         // Add invalidation_key for event-based cache invalidation
         dataToCache.invalidation_key = `checkup:${checkupId}`;

        const compressed = compress_statistics(dataToCache);
        localStorage.setItem(cacheKey, compressed);

        console.log(`💾 Timeline cached for checkup ${checkupId}: ${timelineData.from} → ${timelineData.until}`);
        return true;
    } catch (e) {
        if (e.name === 'QuotaExceededError') {
            console.warn('⚠️ localStorage quota exceeded for timeline cache');
            // Trigger cleanup and retry
            cleanup_timeline_storage();

            try {
                 const dataToCache2 = {...timelineData};
                 if (expiresAt !== null && expiresAt > 0) {
                     dataToCache2.expires_at = expiresAt;
                 }
                const compressed = compress_statistics(dataToCache2);
                localStorage.setItem(cacheKey, compressed);
                console.log(`💾 Timeline cached after cleanup for checkup ${checkupId}`);
                return true;
            } catch (retryError) {
                console.error('❌ Still cannot cache timeline after cleanup:', retryError);
                return false;
            }
        } else {
            console.error('❌ Timeline cache save error:', e);
            return false;
        }
    }
}

/**
 * Loads statistics timeline from cache
 *
 * @param {number} checkupId - Checkup ID
 * @returns {Object|null} Timeline data or null if not found/expired
 */
function load_timeline_cache(checkupId) {
    const cacheKey = `${STATS_CACHE_PREFIX}${checkupId}:timeline`;
    const compressed = localStorage.getItem(cacheKey);

    if (!compressed) {
        console.log(`📭 No timeline cache for checkup ${checkupId}`);
        return null;
    }

    const data = decompress_statistics(compressed);

    if (!data) {
        console.error(`❌ Failed to decompress timeline for checkup ${checkupId}`);
        localStorage.removeItem(cacheKey);
        return null;
    }

    // Check if cache was invalidated via invalidation_key
     if (data.invalidation_key) {
         const invalidationTime = get_cache_invalidation_timestamp(data.invalidation_key);
         if (invalidationTime) {
             const cacheTime = new Date(data.until).getTime() / 1000;
             if (invalidationTime > cacheTime) {
                 console.log(`🗑️ Timeline cache invalidated for checkup ${checkupId} at ${new Date(invalidationTime * 1000).toISOString()}`);
                 localStorage.removeItem(cacheKey);
                 return null;
             }
         }
     }
    console.log(`📦 Timeline loaded for checkup ${checkupId}: ${data.from} → ${data.until}`);
    return data;
}

/**
 * Checks if timeline cache covers requested date range
 *
 * @param {Object} timeline - Timeline cache data
 * @param {string} requestedFrom - Requested start date (ISO format)
 * @returns {Object} Coverage analysis: {covered: boolean, gapBefore: string|null, gapAfter: string|null}
 */
function check_timeline_coverage(timeline, requestedFrom) {
    if (!timeline) {
        return {
            covered: false,
            gapBefore: requestedFrom,
            gapAfter: null
        };
    }

    const cacheFrom = new Date(timeline.from);
    const cacheUntil = new Date(timeline.until);
    const reqFrom = new Date(requestedFrom);
    const now = new Date();

    const gapBefore = reqFrom < cacheFrom ? requestedFrom : null;
    const gapAfter = cacheUntil < now ? true : null; // Always check today for updates

    const covered = !gapBefore && !gapAfter;

    console.log(`🔍 Coverage check: requested=${requestedFrom}, cache=${timeline.from}→${timeline.until}, covered=${covered}`);

    return {
        covered,
        gapBefore,
        gapAfter
    };
}

/**
 * Cleans up timeline storage when quota exceeded
 * Progressively reduces retention for all checkups
 */
function cleanup_timeline_storage() {
    console.log('🧹 Starting timeline storage cleanup...');

    // Get all timeline cache keys
    const keys = [];
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith(STATS_CACHE_PREFIX) && key.endsWith(':timeline')) {
            keys.push(key);
        }
    }

    if (keys.length === 0) {
        console.log('🧹 No timeline caches to clean');
        return;
    }

    console.log(`🧹 Found ${keys.length} timeline cache(s) to potentially reduce`);

    // Try progressive cleanup levels
    for (const level of TIMELINE_RETENTION_LEVELS) {
        console.log(`🧹 Attempting cleanup to ${level.label} retention...`);

        let cleanedCount = 0;

        keys.forEach(key => {
            const compressed = localStorage.getItem(key);
            if (!compressed) return;

            const timeline = decompress_statistics(compressed);
            if (!timeline) return;

            const cutoffDate = new Date();
            if (level.months) {
                cutoffDate.setMonth(cutoffDate.getMonth() - level.months);
            } else if (level.weeks) {
                cutoffDate.setDate(cutoffDate.getDate() - (level.weeks * 7));
            }

            const timelineFrom = new Date(timeline.from);

            // If timeline extends before cutoff, trim it
            if (timelineFrom < cutoffDate) {
                console.log(`🧹 Trimming ${key} from ${timeline.from} to ${cutoffDate.toISOString()}`);

                // TODO: Implement actual trimming logic in Phase 3
                // For now, just remove old caches
                localStorage.removeItem(key);
                cleanedCount++;
            }
        });

        if (cleanedCount > 0) {
            console.log(`🧹 Cleaned ${cleanedCount} timeline(s) to ${level.label} retention`);
            return; // Stop after first successful cleanup level
        }
    }

    console.log('🧹 Cleanup completed');
}

/**
 * Computes daily aggregations from UTC hourly data in user's timezone
 *
 * @param {Object} hourlyData - Hourly data in UTC {"2025-12-20 00:00": {min, max, avg, cnt}}
 * @param {string|null} fromDate - Optional start date filter (ISO format) to exclude hourly data before this time
 * @param {string} dateKey - Date key (YYYY-MM-DD) of the day being processed
 * @returns {Object} Daily data in user timezone {"2025-12-20": {min, max, avg, cnt}}
 *
 * @description
 * Converts hourly UTC data to daily aggregations in user's local timezone.
 * When fromDate is provided, only filters entries on the FIRST day (fromDate's date).
 * All subsequent days are included completely.
 * This ensures partial day counts for time ranges like "last 3 days" from current time. *
 */
function compute_daily_from_hourly_user_tz(hourlyData, fromDate = null, dateKey = null) {
    // Log FIRST and LAST hour keys only to avoid spam
    const sortedKeys = Object.keys(hourlyData).sort();

    // Parse fromDate to timestamp for comparison (only filter first day)
    let fromTimestamp = null;
    let firstHourToSkip = null;
    if (fromDate) {
        // Parse fromDate as UTC, not local time
        // fromDate format: "2025-12-28T15:00" should be interpreted as 15:00 UTC
        const fromDateObj = new Date(fromDate.replace(' ', 'T') + 'Z');
        fromTimestamp = fromDateObj.getTime();

        // Calculate the first hour key that should be skipped
        // fromDate example: "2025-12-28T14:00" → first hour to skip is exactly at fromTimestamp
        firstHourToSkip = fromTimestamp;
    }

    const daily = {};
    let skippedCount = 0;
    let includedCount = 0;

    Object.keys(hourlyData).forEach(utcHourKey => {
        const hourData = hourlyData[utcHourKey];

        // Parse UTC hour key "2025-12-20 00:00" as UTC timestamp
        const utcDateStr = utcHourKey.replace(' ', 'T') + ':00.000Z';
        const utcDate = new Date(utcDateStr);

        // Get local date components (browser timezone = user timezone)
        // toLocaleDateString with 'sv-SE' locale returns YYYY-MM-DD format
        const userDateStr = utcDate.toLocaleDateString('sv-SE');

        // Skip ALL hourly entries BEFORE fromTimestamp, AND skip the FIRST hour AT fromTimestamp
        // This ensures the first (potentially incomplete) hour is always excluded
        if (fromTimestamp && utcDate.getTime() <= firstHourToSkip) {
            skippedCount++;
            return; // Skip this hourly entry
        }

        includedCount++;
        
        if (!daily[userDateStr]) {
            if (hourData.min !== undefined) {
                // Numeric data
                daily[userDateStr] = {
                    min: hourData.min,
                    max: hourData.max,
                    sum: hourData.avg * hourData.cnt,
                    cnt: hourData.cnt
                };
            } else {
                // YesNo data
                daily[userDateStr] = {
                    on: hourData.on || 0,
                    off: hourData.off || 0,
                    cnt: hourData.cnt
                };
            }
        } else {
            if (hourData.min !== undefined) {
                // Numeric data - aggregate
                daily[userDateStr].min = Math.min(daily[userDateStr].min, hourData.min);
                daily[userDateStr].max = Math.max(daily[userDateStr].max, hourData.max);
                daily[userDateStr].sum += hourData.avg * hourData.cnt;
                daily[userDateStr].cnt += hourData.cnt;
            } else {
                // YesNo data - aggregate
                daily[userDateStr].on += hourData.on || 0;
                daily[userDateStr].off += hourData.off || 0;
                daily[userDateStr].cnt += hourData.cnt;
            }
        }
    });

    // Calculate averages for numeric data
    Object.keys(daily).forEach(dateKey => {
        if (daily[dateKey].sum !== undefined) {
            daily[dateKey].avg = daily[dateKey].sum / daily[dateKey].cnt;
            delete daily[dateKey].sum;
        }
    });

    // Return sorted daily object (chronological order)
    // This ensures consistency with API response format
    const sortedDaily = {};
    Object.keys(daily).sort().forEach(dateKey => {
        sortedDaily[dateKey] = daily[dateKey];
    });

    return sortedDaily;
}

// ===================================
// CACHE MANAGEMENT
// ===================================

/**
 * Clears old statistics cache for a checkup
 *
 * @param {number} checkupId - Checkup ID to clean
 * @param {number} [maxAgeDays=90] - Maximum age in days
 * @returns {number} Number of entries removed
 *
 * @description
 * Removes cache entries older than specified days.
 * Default: 90 days retention.
 */
function clear_old_statistics_cache(checkupId, maxAgeDays = STATS_CACHE_MAX_AGE_DAYS) {
    const prefix = `${STATS_CACHE_PREFIX}stats:${checkupId}:`;
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - maxAgeDays);
    const cutoffKey = get_date_key(cutoffDate);

    let removed = 0;
    const keysToRemove = [];

    // Collect keys to remove
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith(prefix)) {
            const dateKey = key.split(':')[3]; // Extract date
            if (dateKey && dateKey < cutoffKey) {
                keysToRemove.push(key);
            }
        }
    }

    // Remove collected keys
    keysToRemove.forEach(key => {
        localStorage.removeItem(key);
        removed++;
    });

    if (removed > 0) {
        console.log(`🗑️ Cleared ${removed} old statistics entries (older than ${maxAgeDays} days)`);
    }

    return removed;
}

/**
 * Clears ALL statistics cache (all checkups)
 *
 * @returns {number} Number of entries removed
 */
function clear_all_statistics_cache() {
    let removed = 0;
    const keysToRemove = [];

    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith(STATS_CACHE_PREFIX)) {
            keysToRemove.push(key);
        }
    }

    keysToRemove.forEach(key => {
        localStorage.removeItem(key);
        removed++;
    });

    console.log(`🗑️ Cleared ${removed} statistics cache entries`);
    return removed;
}

/**
 * Gets cache statistics for debugging
 *
 * @param {number} checkupId - Checkup ID
 * @returns {Object} Cache stats
 */
function get_statistics_cache_info(checkupId) {
    const prefix = `${STATS_CACHE_PREFIX}stats:${checkupId}:`;
    const entries = [];
    let totalSize = 0;
    let totalPoints = 0;

    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith(prefix)) {
            const item = localStorage.getItem(key);
            const data = JSON.parse(item);

            entries.push({
                date: data.date,
                points: data.points.length,
                size: item.length,
                ttl: data.ttl,
                cached_at: data.cached_at
            });

            totalSize += item.length;
            totalPoints += data.points.length;
        }
    }

    entries.sort((a, b) => a.date.localeCompare(b.date));

    return {
        checkupId: checkupId,
        cachedDays: entries.length,
        totalPoints: totalPoints,
        totalSize: totalSize,
        avgSizePerDay: entries.length > 0 ? Math.round(totalSize / entries.length) : 0,
        avgPointsPerDay: entries.length > 0 ? Math.round(totalPoints / entries.length) : 0,
        dateRange: entries.length > 0 ?
            `${entries[0].date} to ${entries[entries.length-1].date}` : 'none',
        oldestEntry: entries[0]?.date,
        newestEntry: entries[entries.length-1]?.date,
        entries: entries
    };
}

/**
 * Gets cache statistics for ALL checkups
 *
 * @returns {Object} Global cache stats
 */
function get_all_statistics_cache_info() {
    const checkups = new Set();
    let totalSize = 0;
    let totalEntries = 0;

    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith(STATS_CACHE_PREFIX)) {
            totalEntries++;
            totalSize += localStorage.getItem(key).length;

            // Extract checkup ID
            const parts = key.split(':');
            if (parts[2]) {
                checkups.add(parseInt(parts[2]));
            }
        }
    }

    return {
        totalCheckups: checkups.size,
        totalCachedDays: totalEntries,
        totalSize: totalSize,
        totalSizeMB: (totalSize / 1024 / 1024).toFixed(2),
        avgSizePerDay: totalEntries > 0 ? Math.round(totalSize / totalEntries) : 0,
        checkupIds: Array.from(checkups).sort((a, b) => a - b)
    };
}

/**
 * Finds continuous gaps in cached dates
 *
 * @param {Array<string>} requestedDates - All requested dates (sorted)
 * @param {Array<string>} cachedDates - Available cached dates
 * @returns {Array<Object>} Array of gap objects with fromDate and tillDate
 *
 * @description
 * Analyzes which date ranges are missing and groups them into
 * continuous gaps for efficient API calls.
 *
 * **Gap Strategy:**
 * - Consolidates missing dates into continuous ranges
 * - Each gap gets fromDate and tillDate for targeted API call
 * - Last gap (if it ends with today) gets no tillDate (open-ended)
 *
 * @example
 * const requested = ['2025-12-11', '2025-12-12', ..., '2025-12-18'];
 * const cached = ['2025-12-15', '2025-12-16', '2025-12-17'];
 * const gaps = find_date_gaps(requested, cached);
 * // Returns: [
 * //   { fromDate: '2025-12-11', tillDate: '2025-12-14' },
 * //   { fromDate: '2025-12-18', tillDate: null }
 * // ]
 */
function find_date_gaps(requestedDates, cachedDates) {
    const missingDates = requestedDates.filter(date => !cachedDates.includes(date));

    if (missingDates.length === 0) {
        return [];
    }

    // Group consecutive missing dates into gaps
    const gaps = [];
    let currentGap = {
        fromDate: missingDates[0],
        tillDate: missingDates[0]
    };

    for (let i = 1; i < missingDates.length; i++) {
        const prevDate = new Date(missingDates[i - 1]);
        const currDate = new Date(missingDates[i]);

        // Check if dates are consecutive (difference = 1 day)
        const dayDiff = (currDate - prevDate) / (1000 * 60 * 60 * 24);

        if (dayDiff === 1) {
            // Extend current gap
            currentGap.tillDate = missingDates[i];
        } else {
            // Start new gap
            gaps.push(currentGap);
            currentGap = {
                fromDate: missingDates[i],
                tillDate: missingDates[i]
            };
        }
    }

    // Add last gap
    gaps.push(currentGap);

    // Optimize: Last gap ending with today should be open-ended (no tillDate)
    const today = get_today_key();
    if (gaps.length > 0 && gaps[gaps.length - 1].tillDate === today) {
        gaps[gaps.length - 1].tillDate = null; // Open-ended to today
    }

    return gaps;
}