/* /lib/js/calendar.js */
/*!
 * Calendar Functions for Website Checkups Plugin v1.2.0
 * https://website-checkups.com
 * Extracted from admin.js and charts.js during refactoring
 * Released under the GPLv2 license
 */
// ===================================
// CALENDAR HELPER FUNCTIONS (reusable)
// ===================================

/**
 * Initializes calendar with event handlers
 *
 * @async
 * @param {string} calendarId - Unique identifier for the calendar instance
 * @param {Array} checkups - Array of checkup objects to display
 * @param {string} timespan - Time period for data aggregation (e.g., 'week', 'month')
 * @param {string} [containerId='calendar_container'] - Container element ID (optional)
 * @returns {Promise<void>}
 *
 * @description
 * Creates a calendar interface showing checkup statistics per day.
 * Fetches template, aggregates checkup data by date, and sets up navigation.
 *
 * @example
 * await initCalendar('main_calendar', checkupList, 'month');
 */
async function initCalendar(calendarId, checkups, timespan, containerId = 'calendar_container') {
    // Load template
    const template = await fetch_template('template_calendar.html');
    if (!template) return;

    const container = getElById(containerId);
    if (!container) return;

    container.innerHTML = template.replaceAll('{{calendar_id}}', calendarId);

    // Show inline navigation for dashboard (if not using external navigation)
    const isDashboard = containerId === 'calendar_container';
    if (isDashboard) {
        const inlineHeader = container.querySelector('.calendar-header-inline');
        if (inlineHeader) {
            inlineHeader.style.display = 'block';
        }
    }

    // Aggregate data by date
    const checkupData = await aggregateCheckupsByDate(checkups, timespan);

    // Display current month
    const today = new Date();
    let currentYear = today.getFullYear();
    let currentMonth = today.getMonth();

    // Initial render
    updateCalendar(calendarId, currentYear, currentMonth, checkupData, timespan);

    // Navigation event handlers
    // Support both inline (dashboard) and external (chart) navigation
    const btnPrev = getElById(calendarId + '_btn_prev') || getElById(calendarId + '_btn_prev_inline');
    const btnNext = getElById(calendarId + '_btn_next') || getElById(calendarId + '_btn_next_inline');

    if (btnPrev) {
        btnPrev.addEventListener('click', function() {
            const state = window.calendar_state[calendarId];
            if (currentMonth === 0) {
                currentMonth = 11;
                currentYear--;
            } else {
                currentMonth--;
            }
            // Update state und verwende state.checkupData statt alte checkupData
            state.currentYear = currentYear;
            state.currentMonth = currentMonth;
            updateCalendar(calendarId, currentYear, currentMonth, state.checkupData, state.timespan);
        });
    }

    if (btnNext) {
        btnNext.addEventListener('click', function() {
            const state = window.calendar_state[calendarId];
            if (currentMonth === 11) {
                currentMonth = 0;
                currentYear++;
            } else {
                currentMonth++;
            }
            // Update state und verwende state.checkupData statt alte checkupData
            state.currentYear = currentYear;
            state.currentMonth = currentMonth;
            updateCalendar(calendarId, currentYear, currentMonth, state.checkupData, state.timespan);
        });
    }

    // Store state for timespan updates
    window.calendar_state = window.calendar_state || {};
    window.calendar_state[calendarId] = {
        currentYear,
        currentMonth,
        checkupData,
        timespan
    };
}

/**
 * Updates calendar display for a given month
 *
 * @param {string} calendarId - Calendar instance identifier
 * @param {number} year - Year to display (e.g., 2024)
 * @param {number} month - Month to display (0-11, where 0 = January)
 * @param {Object} checkupData - Aggregated checkup data by date
 * @param {string} timespan - Time period for filtering
 * @returns {void}
 *
 * @description
 * Renders calendar days, updates month label, and adjusts navigation buttons
 * based on available data range.
 */
function updateCalendar(calendarId, year, month, checkupData, timespan) {
    const timespanRange = getTimespanDateRange(timespan);
    const days = generateCalendarDays(year, month, checkupData, timespanRange);

    // Update month label
    const monthLabel = getElById(calendarId + '_month_label') || getElById(calendarId + '_month_label_inline');
    if (monthLabel) {
        const date = new Date(year, month, 1);
        monthLabel.textContent = date.toLocaleDateString(localLanguage, {
            month: 'long',
            year: 'numeric'
        });
    }

    // Render days
    renderCalendarDays(calendarId, days, timespanRange);

    // Update navigation buttons
    updateCalendarNavigation(calendarId, year, month, timespanRange, checkupData);
}

/**
 * Updates calendar when timespan changes
 *
 * @async
 * @param {string} calendarId - Calendar instance identifier
 * @param {Array} checkups - Array of checkup objects
 * @param {string} newTimespan - New time period to display
 * @returns {Promise<void>}
 *
 * @description
 * Re-aggregates checkup data for new timespan and resets calendar to current month.
 */
async function updateCalendarTimespan(calendarId, checkups, newTimespan) {
    const state = window.calendar_state?.[calendarId];
    if (!state) {
        console.warn('No calendar state found for', calendarId);
        return;
    }

    // console.log('🔄 Updating calendar timespan to:', newTimespan);

    // Re-aggregate data with new timespan
    const checkupData = await aggregateCheckupsByDate(checkups, newTimespan);

    // Reset to current month
    const today = new Date();
    const currentYear = today.getFullYear();
    const currentMonth = today.getMonth();

    // Update state
    state.currentYear = currentYear;
    state.currentMonth = currentMonth;
    state.checkupData = checkupData;
    state.timespan = newTimespan;

    // Render with new data
    updateCalendar(calendarId, currentYear, currentMonth, checkupData, newTimespan);
}

/**
 * Creates month navigation HTML in chart navigation bar
 *
 * @async
 * @param {string} calendarId - Calendar instance identifier
 * @param {string} targetSelector - jQuery selector for navigation container (e.g., '.chart-navigation')
 * @returns {Promise<void>}
 *
 * @description
 * Loads month navigation template and appends it to the target container.
 * Used for calendar views in chart interfaces.
 *
 * @example
 * await createMonthNavigation('details_chart_calendar', '.chart-navigation');
 */
async function createMonthNavigation(calendarId, targetSelector) {
    // Check if already exists
    if (jQuery(targetSelector).find('.month-navigation-wrapper').length > 0) {
        return;
    }

    // Load template
    const template = await fetch_template('misc/template_month_navigation.html');
    if (!template) {
        console.error('Month navigation template not found');
        return;
    }

    // Replace placeholder and append
    const navHtml = template.replace(/\{\{calendar_id\}\}/g, calendarId);
    jQuery(targetSelector).append(navHtml);
}

/**
 * Generates calendar day objects for given month/year
 *
 * @param {number} year - Year to generate (e.g., 2024)
 * @param {number} month - Month to generate (0-11)
 * @param {Object} checkupData - Checkup statistics by date (YYYY-MM-DD format)
 * @param {Object} timespanRange - Object with startDate and endDate properties
 * @returns {Array<Object>} Array of day objects with structure:
 *   {day: number, date: string, isCurrentMonth: boolean, data: Object|null}
 *
 * @description
 * Creates 42 day objects (6 weeks) including trailing/leading days from adjacent months.
 * Each day contains checkup statistics if available.
 */
function generateCalendarDays(year, month, checkupData, timespanRange) {
    const days = [];

    // First day of month
    const firstDay = new Date(year, month, 1);
    const startOfWeek = firstDay.getDay(); // 0 = Sonntag

    // Last day of month
    const lastDay = new Date(year, month + 1, 0);
    const daysInMonth = lastDay.getDate();

    // Previous month (trailing days)
    const prevMonth = month === 0 ? 11 : month - 1;
    const prevYear = month === 0 ? year - 1 : year;
    const daysInPrevMonth = new Date(prevYear, prevMonth + 1, 0).getDate();

    for (let i = startOfWeek - 1; i >= 0; i--) {
        const day = daysInPrevMonth - i;
        const dateStr = `${prevYear}-${String(prevMonth + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;

        days.push({
            day: day,
            date: dateStr,
            isCurrentMonth: false,
            data: checkupData[dateStr] || null
        });
    }

    // Current month
    for (let day = 1; day <= daysInMonth; day++) {
        const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;

        days.push({
            day: day,
            date: dateStr,
            isCurrentMonth: true,
            data: checkupData[dateStr] || null
        });
    }

    // Next month (leading days)
    const remainingDays = 42 - days.length; // 6 weeks * 7 days
    const nextMonth = month === 11 ? 0 : month + 1;
    const nextYear = month === 11 ? year + 1 : year;

    for (let day = 1; day <= remainingDays; day++) {
        const dateStr = `${nextYear}-${String(nextMonth + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;

        days.push({
            day: day,
            date: dateStr,
            isCurrentMonth: false,
            data: checkupData[dateStr] || null
        });
    }

    return days;
}

/**
 * Renders calendar days as HTML
 *
 * @param {string} calendarId - Calendar instance identifier
 * @param {Array<Object>} days - Array of day objects from generateCalendarDays()
 * @param {Object} timespanRange - Date range for highlighting
 * @returns {void}
 *
 * @description
 * Creates HTML elements for each calendar day with appropriate styling based on:
 * - Whether day is in current month
 * - Whether day is within selected timespan
 * - Checkup success/failure statistics
 */
function renderCalendarDays(calendarId, days, timespanRange) {
    const container = getElById(calendarId + '_days');
    if (!container) return;

    const dayElements = [];

    days.forEach(dayData => {
        const classes = ['calendar-day'];

        if (!dayData.isCurrentMonth) {
            classes.push('other-month');
        }

        // Check if day is within timespan
        const inTimespan = isDateInTimespan(dayData.date, timespanRange);

        if (!inTimespan && dayData.isCurrentMonth) {
            classes.push('outside-timespan');
        }

        // Styling based on data
        if (dayData.data && dayData.isCurrentMonth && inTimespan) {
            classes.push('has-data');

            if (dayData.data.failed > 0) {
                classes.push('has-failures');
            } else {
                classes.push('all-success');
            }
        }

        // Stats HTML and Tooltip (different logic for dashboard vs single service)
        let statsHtml = '';
        let tooltipHtml = '';

        if (dayData.data && dayData.isCurrentMonth && inTimespan) {
            statsHtml = '<div class="checkup-stats">';
            if (dayData.data.successful > 0) {
                statsHtml += `<span class="success-count">${formatCalendarNumber(dayData.data.successful)}</span>`;
            }
            if (dayData.data.failed > 0) {
                statsHtml += `<span class="failed-count">${formatCalendarNumber(dayData.data.failed)}</span>`;
            }
            statsHtml += '</div>';

            // Build tooltip based on context (dashboard vs single service)
            if (dayData.data.services && dayData.data.services.length > 0) {
                // Dashboard: Show per-service uptime for first 10 services

                // Sort services: First by uptime (worst first), then by name
                const sortedServices = [...dayData.data.services].sort((a, b) => {
                    const uptimeA = a.total > 0 ? (a.successful / a.total) * 100 : 0;
                    const uptimeB = b.total > 0 ? (b.successful / b.total) * 100 : 0;

                    // Primary: Sort by uptime ascending (worst first)
                    if (uptimeA !== uptimeB) {
                        return uptimeA - uptimeB;
                    }

                    // Secondary: Sort by name alphabetically
                    return a.name.localeCompare(b.name);
                });
                
                tooltipHtml = '<div class="calendar-service-list">';
                const servicesToShow = sortedServices.slice(0, 10);
                servicesToShow.forEach(service => {
                    const uptime = service.total > 0 ? ((service.successful / service.total) * 100).toFixed(1).replace(/\.0$/, '') : '0';
                    // Strukturiertes HTML mit Ellipsis-Support
                    tooltipHtml += `
    <div class="service-item">
        <span class="service-name" title="${service.name.replace(/"/g, '&quot;')}">${service.name}</span>
        <span class="service-uptime">${uptime}%</span>
    </div>
`;
                });
                if (sortedServices.length > 10) {
                    tooltipHtml += `<em>...and ${sortedServices.length - 10} more</em>`;
                }
                tooltipHtml += '</div>';
            } else {
                // Single service (Checkup Results): Show overall uptime and response times
                const total = dayData.data.successful + dayData.data.failed;
                const uptimePercent = total > 0 ? ((dayData.data.successful / total) * 100).toFixed(1).replace(/\.0$/, '') : '0';

                tooltipHtml = '<div style="text-align: left;">';
                tooltipHtml += `<strong>Uptime: ${uptimePercent}%</strong>`;

                if (dayData.data.hasResponseTimes) {
                    tooltipHtml += '<br>---<br>';
                    tooltipHtml += `Min: ${formatNumbers(dayData.data.min)} sec<br>`;
                    tooltipHtml += `Max: ${formatNumbers(dayData.data.max)} sec<br>`;
                    tooltipHtml += `Avg: ${formatNumbers(dayData.data.avg)} sec`;
                }
                tooltipHtml += '</div>';
            }
        }
        // Add tooltip attributes if data exists
        const tooltipAttrs = tooltipHtml
            ? `data-bs-toggle="tooltip" data-bs-placement="top" data-bs-html="true" title="${tooltipHtml.replace(/"/g, '&quot;')}"`
            : '';

        dayElements.push(`
            <div class="${classes.join(' ')}" data-date="${dayData.date}" ${tooltipAttrs}>
                <div class="day-number">${dayData.day}</div>
                ${statsHtml}
            </div>
        `);
    });

    container.innerHTML = dayElements.join('');

    // Initialize Bootstrap tooltips
    if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
        const tooltipElements = container.querySelectorAll('[data-bs-toggle="tooltip"]');
        tooltipElements.forEach(element => {
            new bootstrap.Tooltip(element);
        });
    }
}

/**
 * Updates calendar navigation buttons (Previous/Next)
 *
 * @param {string} calendarId - Calendar instance identifier
 * @param {number} year - Current displayed year
 * @param {number} month - Current displayed month (0-11)
 * @param {Object} timespanRange - Date range with startDate and endDate
 * @param {Object} checkupData - Checkup data (unused, kept for compatibility)
 * @returns {void}
 *
 * @description
 * Controls visibility and state of navigation buttons:
 * - Previous button: Hidden if already at earliest month with data
 * - Next button: Hidden if already at current month
 */
function updateCalendarNavigation(calendarId, year, month, timespanRange, checkupData) {
    // Support both inline (dashboard) and external (chart) navigation
    const btnPrev = getElById(calendarId + '_btn_prev') || getElById(calendarId + '_btn_prev_inline');
    const btnNext = getElById(calendarId + '_btn_next') || getElById(calendarId + '_btn_next_inline');

    if (!btnPrev || !btnNext) return;

    const currentMonthDate = new Date(year, month, 1);

    // Previous button: Can only go back if data exists in previous month
    const minMonthDate = new Date(timespanRange.startDate.getFullYear(), timespanRange.startDate.getMonth(), 1);
    const canGoPrev = currentMonthDate > minMonthDate;

    btnPrev.style.visibility = canGoPrev ? 'visible' : 'hidden';
    btnPrev.disabled = !canGoPrev;

    // Next button: Cannot go beyond current month
    const today = new Date();
    const currentRealMonth = new Date(today.getFullYear(), today.getMonth(), 1);
    const canGoNext = currentMonthDate < currentRealMonth;

    btnNext.style.visibility = canGoNext ? 'visible' : 'hidden';
    btnNext.disabled = !canGoNext;
}

/**
 * Checks if date is within timespan range
 *
 * @param {string} dateStr - Date string in YYYY-MM-DD format
 * @param {Object} timespanRange - Range with startDate and endDate Date objects
 * @returns {boolean} True if date is within range
 */
function isDateInTimespan(dateStr, timespanRange) {
    // Compare date strings directly (YYYY-MM-DD format)
    // Extract date part from startDate and endDate (ignore time)
    const startDateStr = timespanRange.startDate.toISOString().substring(0, 10);
    const endDateStr = timespanRange.endDate.toISOString().substring(0, 10);

    return dateStr >= startDateStr && dateStr <= endDateStr;
}

/**
 * Formats number with thousand separator for calendar (compact)
 *
 * @param {number} num - Number to format
 * @returns {string} Formatted string (e.g., "1.2k" for 1234)
 *
 * @description
 * Uses compact notation for numbers >= 1000 to save space in calendar cells.
 */
function formatCalendarNumber(num) {
    return num >= 1000 ? format_num(num, 0) : num.toString();
}
