/* /lib/js/charts.js */
/*!
 * charts js lib for website checkups plugin v1.2.0
 * https://website-checkups.com
 * Released under the GPLv2 license
 * Use jsmin, YUI Compressor, or other tools to minify
*/
const localTimeZone = Intl.DateTimeFormat().resolvedOptions().timeZone;
const localLang = navigator.language ?? 'en-US';

/**
 * Chart.js plugin to display "No Data" message
 *
 * @description
 * Automatically displays a centered message when all datasets are empty.
 * Registered globally for all Chart.js instances.
 */
Chart.register({
    id: 'noData',
    afterDraw: function (chart) {
        // If there is no data across all datasets, the message will display
        if (
            chart.data.datasets
                .map((d) => d.data.length)
                .reduce((p, a) => p + a, 0) === 0
        ) {
            // No data is present
            const ctx = chart.ctx;
            const width = chart.width;
            const height = chart.height;
            chart.clear();
            ctx.save();

            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.font = '2rem Helvetica';
            ctx.fillStyle = '#0079a3';
            ctx.fillText('No Checkups Data to display', width / 2, height / 2);
            ctx.restore();
        }
    },
});

/**
 * Determines appropriate time unit for chart X-axis
 *
 * @param {Date|string} start - Start date
 * @param {Date|string} end - End date
 * @returns {string} Time unit: 'year', 'quarter', 'month', 'week', 'day', 'hour', or 'minute'
 *
 * @description
 * Calculates optimal time unit based on date range:
 * - > 1000 days: 'year'
 * - > 365 days: 'quarter'
 * - > 125 days: 'month'
 * - > 30 days: 'week'
 * - >= 1.2 days: 'day'
 * - > 0.2 days (4.8 hours): 'hour'
 * - Otherwise: 'minute'
 */
function getDateIntervalForChart(start, end)
{
    const startDate = new Date(start);
    const endDate = new Date(end);
    const timeDiffDays = new Date(endDate - startDate) / 1000 / 60 / 60 / 24;

    let result = '';
    if (timeDiffDays > 1000) {
        result = 'year';
    } else if (timeDiffDays > 365) {
        result = 'quarter';
    } else if (timeDiffDays > 125) {
        result = 'month';
    } else if (timeDiffDays > 30) {
      result = 'week';
    } else if (timeDiffDays >= 1.2) {
        result = 'day';
    } else if (timeDiffDays > 0.2) {
        result = 'hour';
    } else {
        result = 'minute';
    }
//    console.log('time', result);

    return result;
}

/**
 * Creates doughnut/pie chart for overview statistics
 *
 * @param {string} elementId - Container element ID
 * @param {string} type - Chart type: 'overview' (services) or 'checkups'
 * @param {Array<string>} title - Chart title (can be multi-line array)
 * @param {number} success - Count of successful items
 * @param {number} warning - Count of warning items (only for 'overview' type)
 * @param {number} failed - Count of failed items
 * @returns {void}
 *
 * @description
 * Creates Chart.js doughnut chart with pre-configured styles:
 *
 * **Overview type (services):**
 * - 3 segments: Healthy, Warnings, Critical
 * - Fixed size (70% radius, non-responsive)
 * - Use for service status overview
 *
 * **Checkups type:**
 * - 2 segments: Successful, Failed
 * - Responsive (60% radius)
 * - Use for checkup results
 *
 * Colors: Green (healthy), Yellow (warnings), Red (failed/critical)
 */
function makePieChart(elementId, type, title, success, warning, failed) {
// jQuery('#' + elementId).find('canvas.overview').remove(); // this is my <canvas> element
    const settings = type == 'overview' ? {
        info: [
            'running without issues',
            'running with non-critical issues',
            'currently down or with critical errors',
        ],
        labels: [
            'Healthy ✅',
            'Warnings ⚠️',
            'Offline / Critical ❌',
        ],
        colors: [
            'rgba(0, 255, 0, 0.75)',
            'rgba(255, 193, 7, 0.75)',
            'rgb(225, 50, 50)'
        ],
        data: [success, warning, failed],
        name_sg: ' service is ',
        name_pl: ' services are ',
        label: ' Services',
        radius: '70%',
        responsive: false
    } : {
        info: [
            'completed without issues',
//            'with non-critical issues',
            'unsuccessful',
        ],
        labels: [
            'Successful ✅',
//            'With Warnings ⚠️',
            'Failed ❌',
        ],
        colors: [
            'rgba(0, 255, 0, 0.75)',
            'rgb(225, 50, 50)'
        ],
        data: [success, failed],
        name_sg: ' checkup was ',
        name_pl: ' checkups were ',
        label: ' Checkups',
        radius: '60%',
        responsive: true
    };
    const overviewCanvasId = createCanvas(
        elementId,
        'doughnut',
        upperFirst(settings.label.trim()) + ' Overview'
    );
    const ctx = document.getElementById(overviewCanvasId);

    new Chart(ctx,
        {
            type: 'doughnut',
            radius: settings.radius,
            data: {
                labels: settings.labels,
                datasets: [{
                    label: settings.label,
                    data: settings.data,
                    backgroundColor: settings.colors,
                    hoverOffset: 10
                }]
            },
            options: {
                responsive: settings.responsive,
                plugins: {
                    title: {
                        display: true,
                        text: title,
                        font: {
                            size: 15,
                        },
                        padding: {
                            bottom: 10,
                        },
                    },
                    tooltip: {
                        enabled: true,
                        titleFontSize: 20,
                        bodyFontSize: 19,
                        titleAlign: 'center',
                        position: 'nearest',
                        callbacks: {
                            label: function(tooltipItem, data) {
                                const ind = tooltipItem.dataIndex;
                                const name = tooltipItem.parsed > 1 ? settings.name_pl : settings.name_sg;
                                return ' ' + formatNumbers(tooltipItem.parsed, 0) + name + settings.info[ind] + ' ';
                            },
                        }
                    },
                }
            },
        }
    );
}

/**
 * Generates random integer between min and max (inclusive)
 *
 * @param {number} min - Minimum value
 * @param {number} max - Maximum value
 * @returns {number} Random integer
 */
function getRandomInt(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
}

/**
 * Builds chart navigation UI (type dropdown + granularity buttons)
 *
 * @async
 * @param {string} elementId - Container element ID
 * @param {Object} navigationConfig - Navigation configuration
 * @param {Array<Object>} navigationConfig.chartTypes - Available chart types
 * @param {Array<string>} navigationConfig.availableGranularities - Available granularities
 * @param {string} navigationConfig.currentGranularity - Currently active granularity
 * @param {boolean} navigationConfig.showGranularityControls - Show/hide granularity controls
 * @param {boolean} navigationConfig.hideChartTypeSelector - Hide chart type dropdown completely
 * @returns {Promise<void>}
 *
 * @description
 * Creates interactive navigation controls for charts:
 *
 * **Chart Type Dropdown:**
 * - Lists available chart types (availability, doughnut, failures)
 * - Pre-selects default type
 * - Can be hidden completely (dashboard slider)
 *
 * **Granularity Buttons:**
 * - Daily / Hourly / Realtime options
 * - Shows only available granularities
 * - Highlights active granularity
 * - Hides completely if only one option available
 * - Only visible for 'availability' chart type
 *
 * Uses template: template_chart_navigation.html
 * Initializes Bootstrap tooltips if available
 */
async function buildChartNavigation(elementId, navigationConfig) {
    const {
        chartTypes,
        availableGranularities,
        currentGranularity,
        showGranularityControls,
        hideChartTypeSelector
    } = navigationConfig;

    // Load template
    const template = await fetch_template('template_chart_navigation.html');
    if (!template) {
        console.error('Chart navigation template not found');
        return;
    }

    const dataUid = 'nav_' + Date.now();
    let navigationHtml = template.replace(/\{\{data_uid\}\}/g, dataUid);

    // Find or create container
    const container = jQuery('#' + elementId);
    let navContainer = container.find('.chart-navigation');

    if (navContainer.length === 0) {
        container.prepend(navigationHtml);
        navContainer = container.find('.chart-navigation');
    }

    // Hide chart type selector if requested (dashboard slider)
    const chartTypeWrapper = navContainer.find('.chart-type-selector-wrapper');
    if (hideChartTypeSelector) {
        chartTypeWrapper.hide();
    } else {
        chartTypeWrapper.show();
    }
    
    // Populate chart type dropdown
    const dropdown = navContainer.find('#dropdown_chart_type');
    dropdown.empty();

    chartTypes.forEach(type => {
        const option = jQuery('<option>')
            .val(type.value)
            .text(type.label)
            .prop('selected', type.selected);
        dropdown.append(option);
    });

    // Granularity controls setup
    const granularityWrapper = navContainer.find('.granularity-controls-wrapper');
    const buttons = granularityWrapper.find('.granularity-btn');

    const hasMultipleOptions = availableGranularities.length > 1;
    const shouldShow = showGranularityControls && hasMultipleOptions;

    if (shouldShow) {
        granularityWrapper.show();

        // Update buttons
        buttons.each(function() {
            const btn = jQuery(this);
            const granularity = btn.data('granularity');
            const isAvailable = availableGranularities.includes(granularity);
            const isActive = granularity === currentGranularity;

            if (isAvailable) {
                btn.show()
                    .prop('disabled', false)
                    .toggleClass('active', isActive);
            } else {
                btn.hide();
            }
        });

        // Initialize Bootstrap tooltips
        if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
            buttons.each(function() {
                const tooltip = new bootstrap.Tooltip(this);
                // Remove tooltip on mouseout, click and scroll
                jQuery(this).on('mouseout click', function() {
                    tooltip.hide();
                });
                // Remove tooltip on scroll
                jQuery(window).on('scroll', function() {
                    tooltip.hide();
                });
            });
        }
    } else {
        granularityWrapper.hide();
    }
}

/**
 * Main function to draw all statistics charts
 *
 * @async
 * @param {Object} data - API response data with statistics
 * @param {string} elementId - Container element ID
 * @param {boolean} [showAll=true] - Show all chart types (availability, overview, failures)
 * @param {string|null} [granularityOverride=null] - Manual granularity override
 * @param {boolean} [disableGranularityControls=false] - If true, hides granularity controls completely
 * @param {boolean} [hideChartTypeSelector=false] - If true, hides chart type dropdown completely
 * @returns {Promise<void>}
 *
 * @description
 * Orchestrates creation of all chart types based on available data.
 *
 * **Process:**
 * 1. Determines optimal data variant (granularity)
 * 2. Prepares success and failure data
 * 3. Calculates statistics (min/max, uptime, etc.)
 * 4. Creates availability chart (if success data exists)
 * 5. Creates overview doughnut chart (if showAll=true)
 * 6. Creates failures chart (if failure data exists)
 * 7. Builds navigation UI with chart type selector
 *
 * **Chart types created:**
 * - Availability: Line/scatter chart showing response times or availability
 * - Overview: Doughnut chart with success/failure summary
 * - Failures: Bar chart showing failure distribution over time
 *
 * **Granularity controls:**
 * - Shown only when: !disableGranularityControls AND availableGranularities.length > 1
 * - Hidden on dashboard slider for cleaner small charts
 *
 * **Chart type selector:**
 * - Hidden when: hideChartTypeSelector=true (dashboard slider)
 * - Shown on details page with localStorage persistence
 *
 * **Global variables set:**
 * - webcheckups_use_optimized_scale
 * - webcheckups_optimized_min/max
 * - webcheckups_shows_all_data
 *
 * These are used by chart refresh functionality.
 */
async function drawStatGraphic(data, elementId, showAll=true, granularityOverride=null, disableGranularityControls=false, hideChartTypeSelector=false) {
    // Prepare graphic data
    const variantData = getVariant(data, granularityOverride);
//    console.log('variantData', variantData);
    let graphDataOk = prepareOkData(data, variantData);
    let graphDataFail = prepareFailData(data);
    const service = getServiceType(data?.meta?.info?.requestAs);

//    console.log('graphDataOk_', graphDataOk);
//    console.log('graphDataFail', graphDataFail);
    const serviceName = data?.meta?.checkup?.url || data?.meta?.checkup?.name || '';

    const minMaxValues = getMinMaxValues(graphDataOk, graphDataFail);

    // Store optimized Y-axis values in global variables
    window.webcheckups_use_optimized_scale = !variantData.isYesNo;
    window.webcheckups_optimized_min = minMaxValues.optimizedMin;
    window.webcheckups_optimized_max = minMaxValues.optimizedMax;
    window.webcheckups_shows_all_data = minMaxValues.showsAllData;

    jQuery('#' + elementId).find('canvas').remove();
    jQuery('#' + elementId).find('div').not('.calendar-chart-wrapper, .month-navigation-wrapper').remove();

    const chartTypes = [];
    // Load last selected chart type from localStorage (only if NOT on dashboard)
    let savedChartType = null;
    if (!hideChartTypeSelector && typeof localStorage !== 'undefined') {
        savedChartType = localStorage.getItem('webcheckups_last_chart_type');
    }

    const upTimePercent = ' ' + formatNumbers(minMaxValues.uptimePercent, 2).
        toString().replace('.00', '').replace(',00', '') + '%';
    const titleSummary = serviceName ? 'Availability Summary - ' + serviceName : upperFirst(service) + ' Availability';
    const titleTimeChart = 'Availability' +
        (!variantData.isYesNo && minMaxValues?.avg ? ' & Response Time' : '') +
        (serviceName ? ' - ' + serviceName : ' - ' + service);
    const titleStat = 'Uptime: ' + upTimePercent +
        (!variantData.isYesNo && minMaxValues?.avg ? ' | Avg. Response: ' + formatNumbers(minMaxValues.avg) + ' sec' : '');

//    console.log('minMaxVal', minMaxValues);
    if (graphDataOk.length > 0 || graphDataFail.length == 0) {
        let minValue = minMaxValues.min;
        let maxValue = minMaxValues.max;
        createSuccessGraphic(
            graphDataOk,
            createCanvas(elementId, 'availability', service),
            [titleTimeChart, titleStat],
            minValue,
            maxValue,
            variantData,
            service
        );

        chartTypes.push({
            value: 'availability',
            label: 'Availability',
            selected: !savedChartType || savedChartType === 'availability'
        });
    }

    if (showAll && (graphDataOk.length > 0 || graphDataFail.length > 0)) {
        makePieChart(elementId, 'checkups', [titleSummary, titleStat], minMaxValues.total, 0, minMaxValues.failed);

        chartTypes.push({
            value: 'doughnut',
            label: 'Overview',
            selected: savedChartType === 'doughnut'
        });
    }

    if (graphDataFail.length > 0 && (showAll || graphDataOk.length == 0)) {
        // output errors in table, if div for it exists
        const failedData = prepareFailedData(graphDataFail);
//        console.log('failedData', failedData);

        let graphicTitle = serviceName ?
            'Failures ' + serviceName + ' ' + formatNumbers(failedData.total, 0) :
            upperFirst(service) + ' failures: ' + formatNumbers(failedData.total, 0);
        const failedCanvasId = createCanvas(elementId, 'failures', service);
        createFailedGraphic(
            failedData.graphData,
            failedCanvasId,
            graphicTitle,
            0,
            failedData.max + 1,
            failedData.unit,
            service
        );

        chartTypes.push({
            value: 'failures',
            label: 'Failures',
            selected: savedChartType === 'failures'
        });
    }

    // Calendar view (always available if we have data)
    if (graphDataOk.length > 0 || graphDataFail.length > 0) {
        // Create calendar container
        const calendarContainer = document.createElement('div');
        calendarContainer.id = elementId + '_calendar_wrapper';
        calendarContainer.className = 'calendar-chart-wrapper';
        calendarContainer.style.display = 'none';
        jQuery('#' + elementId).append(calendarContainer);

        chartTypes.push({
            value: 'calendar',
            label: 'Calendar',
            selected: savedChartType === 'calendar'
        });
    }
    
    // hide all non-selected charts
    jQuery('#' + elementId).find('canvas').each(function() {
        const canvas = jQuery(this);
        const canvasClasses = canvas.attr('class') || '';
        const isSelected = chartTypes.find(ct =>
            ct.selected && canvasClasses.includes(ct.value)
        );
        canvas.toggle(!!isSelected);
    });
    
    const navigationConfig = {
        chartTypes: chartTypes,
        availableGranularities: variantData.availableGranularities,
        currentGranularity: variantData.variant,
        showGranularityControls: !disableGranularityControls && variantData.availableGranularities.length > 1,
        hideChartTypeSelector: hideChartTypeSelector
    };
    // Create navigation
    await buildChartNavigation(elementId, navigationConfig);

    // Chart type dropdown change handler
    // Only attach handler if chart type selector is visible
    if (!hideChartTypeSelector) {
        jQuery('#dropdown_chart_type').off('change').on('change', async function () {
            const selectedType = jQuery(this).val();
            // Save selection to localStorage
            localStorage.setItem('webcheckups_last_chart_type', selectedType);

            const granularityWrapper = jQuery('#' + elementId).find('.granularity-controls-wrapper');
            const calendarWrapper = jQuery('#' + elementId + '_calendar_wrapper');

            jQuery('#' + elementId).find('canvas').each(function () {
                const canvas = jQuery(this);
                const canvasClasses = canvas.attr('class') || '';
                canvas.toggle(canvasClasses.includes(selectedType));
            });

            // Handle calendar visibility
            if (selectedType === 'calendar') {
                // console.log('📅 Calendar selected, wrapper exists:', calendarWrapper.length > 0);
                calendarWrapper.addClass('active').show();

                // Initialize calendar if empty
                if (calendarWrapper.find('.calendar-chart').length === 0) {
                    // console.log('📅 Initializing calendar for first time');
                    // Get checkupId from dropdown
                    const checkupDropdown = document.getElementById('dropdown_checkup_list');
                    const checkupId = checkupDropdown ? parseInt(checkupDropdown.value) : null;
                    // console.log('📅 Checkup ID:', checkupId);

                    if (checkupId) {
                        // Get timespan from dropdown
                        const periodDropdown = document.getElementById('dropdown_checkup_result_period');
                        const timespan = periodDropdown ? periodDropdown.value : 'week';
                        // console.log('📅 Timespan:', timespan);

                        const calendarId = elementId + '_calendar';

                        // Create month navigation FIRST (before initCalendar needs it)
                        await createMonthNavigation(calendarId, '#' + elementId + ' .chart-navigation');

                        // Get name from dropdown text
                        const checkupName = checkupDropdown.options[checkupDropdown.selectedIndex]?.text || 'Service';
                        const checkupForCalendar = [{
                            checkupId: checkupId,
                            name: checkupName
                        }];
                        // console.log('📅 Calling initCalendar with:', elementId + '_calendar', checkupForCalendar, timespan);

                        // Initialize calendar first, then load month navigation
                        try {
                            await initCalendar(elementId + '_calendar', checkupForCalendar, timespan, elementId + '_calendar_wrapper');
                            // console.log('📅 Calendar initialized successfully');
                        } catch (err) {
                            console.error('📅 Calendar initialization failed:', err);
                        }
                    } else {
                        console.error('📅 No checkup ID found in data');
                    }
                } else {
                    // console.log('📅 Calendar already initialized');
                    // Calendar exists, but we need to reload data for new service/period
                    const checkupDropdown = document.getElementById('dropdown_checkup_list');
                    const checkupId = checkupDropdown ? parseInt(checkupDropdown.value) : null;

                    if (checkupId) {
                        const periodDropdown = document.getElementById('dropdown_checkup_result_period');
                        const timespan = periodDropdown ? periodDropdown.value : 'week';
                        const checkupName = checkupDropdown.options[checkupDropdown.selectedIndex]?.text || 'Service';

                        const checkupForCalendar = [{
                            checkupId: checkupId,
                            name: checkupName
                        }];

                        // console.log('📅 Reloading calendar data for checkup:', checkupId, 'timespan:', timespan);

                        // Re-aggregate data and update calendar
                        const checkupData = await aggregateCheckupsByDate(checkupForCalendar, timespan);

                        // Get current month from state or use today
                        const calendarId = elementId + '_calendar';
                        const state = window.calendar_state?.[calendarId];
                        const currentYear = state?.currentYear || new Date().getFullYear();
                        const currentMonth = state?.currentMonth || new Date().getMonth();

                        // Update state with new data
                        if (state) {
                            state.checkupData = checkupData;
                            state.timespan = timespan;
                        }

                        // Re-render calendar with new data
                        updateCalendar(calendarId, currentYear, currentMonth, checkupData, timespan);
                        // console.log('📅 Calendar data reloaded');
                    }

                    const monthNavigation = jQuery('#' + elementId).find('.month-navigation-wrapper');
                    monthNavigation.show();
                }
            } else {
                calendarWrapper.removeClass('active').hide();
                const monthNavigation = jQuery('#' + elementId).find('.month-navigation-wrapper');
                monthNavigation.hide();
            }

            // how granularity only for availability chart
            if (selectedType === 'availability') {
                granularityWrapper.show();
            } else {
                granularityWrapper.hide();
            }
        });
    }
    // Trigger initial change to set up visibility correctly
    jQuery('#dropdown_chart_type').trigger('change');
}

/**
 * Enlarges timespan by one hour in either direction
 *
 * @param {Date|string} date - Date to enlarge
 * @param {boolean} [add=true] - If true, add 1 hour; if false, subtract 1 hour
 * @returns {string} ISO date string (YYYY-MM-DDTHH:mmZ format)
 *
 * @description
 * Used to add padding to chart date ranges for better visualization.
 */
function enlargeTimespan(date, add=true) {
    let d = new Date(date);
    d.setHours(d.getHours() + (add ? 1 : -1));
    return getCuttedDateString(d);
}

/**
 * Finds minimum and maximum dates from graph data array
 *
 * @param {Array<Object>} graphData - Array of data points with x or performed property
 * @returns {Object} {minDate: string, maxDate: string} or {minDate: undefined, maxDate: undefined}
 *
 * @description
 * Extracts date range from first and last data points.
 *
 * **Special handling:**
 * If timespan < 3 hours, enlarges range by 1 hour on each side
 * for better chart visualization.
 *
 * Returns undefined dates if array is empty.
 */
function getMinMaxDateFromObject(graphData) {
    let result = {minDate: null, maxDate: null};
    if (graphData.length == 0) {
        return {minDate: undefined, maxDate: undefined};
    }
    const date1 = graphData[0]?.x ?? graphData[0]?.performed;
    const date2 = graphData[graphData.length - 1]?.x ?? graphData[graphData.length - 1]?.performed;

    const dateDiff = (new Date(date1).getTime()) - (new Date(date2).getTime());
    result.minDate = dateDiff < 0 ? date1 : date2;
    result.maxDate = dateDiff < 0 ? date2 : date1;

    // less than 3 hours - make small enlargement
    if (Math.abs(dateDiff) < (3 * 3600 * 1000) ) {
        result.minDate = enlargeTimespan(result.minDate, false);
        result.maxDate = enlargeTimespan(result.maxDate, true);
    }

    return result;
}

/**
 * Creates canvas element for chart rendering
 *
 * @param {string} elementId - Parent container ID
 * @param {string} type - Chart type (used as CSS class and ID prefix)
 * @param {string} service - Service type name (for accessibility label)
 * @returns {string} Generated canvas element ID
 *
 * @description
 * Creates accessible canvas element with:
 * - Unique random ID
 * - Type-based CSS class
 * - ARIA label for accessibility
 * - Fallback text for non-canvas browsers
 */
function createCanvas(elementId, type, service) {
    const randomId = getRandomInt(1, 100000);
    const label = upperFirst(service) + ' ' + upperFirst(type);
    const canvasId = 'graphicdata_' + type.substring(0, 1) + randomId;

    jQuery('#' + elementId).append('<canvas id="' + canvasId + '" class="' + type + '" aria-label="' + label + '" role="img">\n' +
        '  <p>Your browser does not support the canvas element.</p>\n' +
        '</canvas>');
    return canvasId;
}

/**
 * Creates bar chart for checkup failures
 *
 * @param {Array<Object>} graphData - Array of failure data points
 * @param {string} elementId - Canvas element ID
 * @param {Array<string>} title - Chart title (can be multi-line)
 * @param {number} minValue - Minimum Y-axis value
 * @param {number} maxValue - Maximum Y-axis value
 * @param {string} unit - Time unit for X-axis ('minute', 'hour', 'day', etc.)
 * @param {string} service - Service type name
 * @returns {void}
 *
 * @description
 * Creates Chart.js bar chart showing failure distribution over time.
 *
 * **Features:**
 * - Time-series X-axis with adaptive formatting
 * - Integer-only Y-axis (failure counts)
 * - Red bars for failures
 * - Detailed tooltip with:
 *   - Total failures in time bucket
 *   - First/last failure timestamps
 *   - Most frequent error message
 *
 * **Time formatting:**
 * - Long range (> week): Month/day
 * - Medium range (day-week): Day/month
 * - Short range (< day): Hour:minute
 */
function createFailedGraphic(graphData, elementId, title, minValue, maxValue, unit, service)
{
    const label = 'Webservice checkup failures';

    const dates = getMinMaxDateFromObject(graphData);
    let minDate =  new Date(dates.minDate);
    let maxDate = new Date(dates.maxDate);
    const errGraphType = unit === 'minute' || unit === 'hour' ? 'errHourly' : 'errDaily';
    const minDateFormatted = minDate.toLocaleDateString(localLang);
    const maxDateFormatted = maxDate.toLocaleDateString(localLang);

    const ctx = document.getElementById(elementId);
//
//    console.log('unit', unit);
//    console.log('failedData', graphData);
    new Chart(ctx,
        {
            type: 'bar',
            data: {
                datasets: [{
                    data: graphData,
                    maxBarThickness: 6,
                    fill: true,
                    label: label,
                    backgroundColor: 'rgb(225, 50, 50)',
                }],
            },
            options: {
                responsive: true,
                scales: {
                    x: {
                        min: minDate,
                        max: maxDate,
                        ticks: {
                            callback: function(value)
                            {
                                let D = new Date(value);
                                return ['year', 'quarter', 'month', 'week'].includes(unit) ?
                                    D.toLocaleDateString(localLang, {day: 'numeric', month: 'short'}) :
                                    'day' == unit ?
                                        D.toLocaleString(localLang, {day: 'numeric', month: 'short'}) :
                                        D.toLocaleTimeString(localLang, {timeStyle: 'short'});
                            },
                        },
                        type: 'timeseries',
                        time: {
                            unit: unit !== 'minute' ? unit : 'hour',
                            round: true,
                            displayFormats: {
                                minute: 'mm',
                                hour: 'HH',
                                day: 'MMM DD',
                            },
                        },
                        title: {
                            display: true,
                            text: (minDateFormatted == maxDateFormatted ? 'Date: ' + minDateFormatted :
                                'Date Range: ' + minDateFormatted + ' - ' + maxDateFormatted),
                        },
                    },
                    y: {
                        min: minValue,
                        max: maxValue,
                        ticks: {
                          stepSize: 1,
                        },
                        title: {
                            display: true,
                            text: 'Failed Checkups',
                        },
                    }
                },
                hover: {
                    mode: 'nearest',
                    intersect: true
                },
                interaction: {
                    mode: 'x',
                    intersect: true,
                },
                plugins: {
                    title: {
                        display: true,
                        text: title,
                        font: {
                            size: 15,
                        },
                        padding: {
                            bottom: 10,
                        },
                    },
                    tooltip: {
                        enabled: true,
                        titleFontSize: 20,
                        bodyFontSize: 19,
                        titleAlign: 'center',
                        position: 'nearest',
                        callbacks: {
                            title: function(tooltipItems) {
                                let res = '';
                                tooltipItems.forEach(function(tooltipItem) {
                                    if (tooltipItem.raw?.x) {
                                        res = makeTitle(tooltipItem.raw, errGraphType);
                                    }
                                });
                                return res;
                            },
                            label: function(tooltipItem)
                            {
                                const res = [' ' + upperFirst(service) + ' Failed Checkups: '];
                                res.push('');
                                res.push(' Total Failures: ' + formatNumbers(tooltipItem.raw?.y, 0));
                                if (tooltipItem.raw?.y > 1) {
                                    res.push(' First Failure Detected: ' + printHoursMinutes(tooltipItem.raw?.from, true));
                                    res.push(' Last Failure Detected: ' + printHoursMinutes(tooltipItem.raw?.until, true));
                                } else {
                                    res.push(' Time of Checkup: ' + printHoursMinutes(tooltipItem.raw?.from));
                                }
                                if (tooltipItem.raw?.y > tooltipItem.raw?.errSeq) {
                                    if (tooltipItem.raw?.errSeq > 1) {
                                        res.push('Most Frequent Error (' + tooltipItem.raw?.errSeq + ' occurrences):');
                                    } else {
                                        res.push('First Recorded Error:');
                                    }
                                } else {
                                    res.push('Error Message:');
                                }
                                res.push('');
                                const arrErrors = splitStringInChunks(tooltipItem.raw?.error);
                                for (let i = 0; i < arrErrors.length; i++) {
                                    res.push(arrErrors[i]);
                                }

                                return res;
                            },
                        }
                    },
                }
            }
        }
    );
}

/**
 * Calculates optimal line/bar thickness based on data density
 *
 * @param {Object} variantData - Variant configuration object
 * @param {number} variantData.count - Number of data points
 * @param {boolean} variantData.isYesNo - Whether in YesNo (binary) mode
 * @returns {number} Thickness value in pixels
 *
 * @description
 * Dynamically adjusts chart element thickness for optimal readability.
 *
 * **Base thickness (YesNo mode):**
 * - <= 25 points: 4px
 * - <= 50 points: 3px
 * - <= 100 points: 2px
 * - > 100 points: 1px
 *
 * **Numeric mode:** Base thickness × 1.5 (thicker for better visibility)
 *
 * Prevents overcrowding in dense datasets and ensures visibility in sparse ones.
 */
function getLineThickness(variantData) {
    const count = variantData.count;
    const baseValue = count <= 25 ? 4 : count <= 50 ? 3 : count <= 100 ? 2 : 1;
    return variantData.isYesNo ? baseValue : baseValue * 1.5;
}

/**
 * Calculates optimal point radius based on data density
 *
 * @param {Object} variantData - Variant configuration object
 * @param {number} variantData.count - Number of data points
 * @returns {number} Point radius in pixels
 *
 * @description
 * Dynamically adjusts scatter plot point size for optimal visibility.
 *
 * **Radius scale (adaptive):**
 * - <= 10 points: 7px (very visible)
 * - <= 15 points: 6px
 * - <= 20 points: 5px
 * - <= 25 points: 4.5px
 * - <= 37 points: 4px
 * - <= 50 points: 3.5px
 * - <= 75 points: 3.25px
 * - <= 100 points: 2.75px
 * - <= 125 points: 2.25px
 * - > 125 points: 1.75px (prevent overcrowding)
 *
 * Primarily used for YesNo scatter plots to ensure points remain visible
 * regardless of data density.
 */
function getPointRadius(variantData)
{
    const cnt = variantData.count;
    return cnt <= 10 ? 7 : cnt <= 15 ? 6 : cnt <= 20 ? 5 : cnt <= 25 ? 4.5 : cnt <= 37 ? 4 :
        cnt <= 50 ? 3.5 : cnt <= 75 ? 3.25 : cnt <= 100 ? 2.75 : cnt <= 125 ? 2.25 : 1.75;
}

/**
 * Creates availability/response time chart
 *
 * @param {Array<Object>} graphData - Array of success data points
 * @param {string} elementId - Canvas element ID
 * @param {Array<string>} title - Chart title (can be multi-line)
 * @param {number} minValue - Minimum Y-axis value
 * @param {number} maxValue - Maximum Y-axis value
 * @param {Object} variantData - Variant configuration
 * @param {string} service - Service type name
 * @returns {void}
 *
 * @description
 * Creates Chart.js line/scatter chart for successful checkups.
 *
 * **Two modes:**
 *
 * **YesNo Mode (binary availability):**
 * - Type: Scatter plot
 * - Y-axis: Categorical (Yes/No)
 * - Colors: Green (Yes), Red (No)
 * - Point style: Rectangles
 * - Shows availability status only
 *
 * **Numeric Mode (response times):**
 * - Type: Bar chart
 * - Y-axis: Numeric (seconds)
 * - Color: Green bars
 * - Shows actual response times
 * - Tooltip includes min/max/avg for aggregated data
 *
 * **Adaptive point size:**
 * Based on data density (fewer points = larger markers)
 *
 * **Time formatting:**
 * Automatically adapts based on date range (see getDateIntervalForChart)
 */
function createSuccessGraphic(graphData, elementId, title, minValue, maxValue, variantData, service)
{
    const dates = getMinMaxDateFromObject(graphData);
    const variant = variantData.variant;
    const isYesNo = variantData.isYesNo;

    const minDate = new Date(dates.minDate);
    const maxDate = new Date(dates.maxDate);
    const minDateFormatted = minDate.toLocaleDateString(localLang);
    const maxDateFormatted = maxDate.toLocaleDateString(localLang);
    const graphUnit = getDateIntervalForChart(minDate, maxDate);

    const ctx = document.getElementById(elementId);
//    console.log('graphData', graphData);
    const thickness = getLineThickness(variantData);
    new Chart(ctx,
        {
            type: isYesNo ? 'scatter' : 'bar',
            data: {
                datasets: isYesNo ? [{
                    data: graphData.filter(e => e.y == 'Yes'),
                    maxBarThickness: thickness,
                    fill: true,
                    label: 'Success',
                    backgroundColor: 'rgba(0, 255, 0, 0.75)',
                }, {
                    data: graphData.filter(e => e.y == 'No'),
                    maxBarThickness: thickness,
                    fill: true,
                    label: 'Failures',
                    backgroundColor: 'rgb(225, 50, 50)',
                }] : [{
                    data: graphData,
                    maxBarThickness: thickness,
                    fill: true,
                    label: upperFirst(service) + ' performance',
                    backgroundColor: 'rgba(0, 255, 0, 0.75)',
                }],
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                elements: isYesNo ? { point: {pointStyle: 'rect', radius: getPointRadius(variantData)}} : {},
                scales: {
                    x: {
                        min: minDate,
                        max: maxDate,
                        ticks: {
                            callback: function(value)
                            {
                                let D = new Date(value);
                                return ['year', 'quarter', 'month', 'week'].includes(graphUnit) ?
                                    D.toLocaleDateString(localLang, {day: 'numeric', month: 'short'}) :
                                    'day' == graphUnit ?
                                    D.toLocaleString(localLang, {day: 'numeric', month: 'short'}) :
                                    D.toLocaleTimeString(localLang, {timeStyle: 'short'});
                            },
                        },
                        type: 'timeseries',
                        time: {
                            unit: graphUnit,
                            round: true,
                            displayFormats: {
                                hour: 'HH',
                                day: 'DD ',
                            },
                        },
                        title: {
                            display: true,
                            text: (minDateFormatted == maxDateFormatted ? 'Date: ' + minDateFormatted :
                                'Date Range: ' + minDateFormatted + ' - ' + maxDateFormatted),
                        },
                    },
                    y: isYesNo ? {
                        type: 'category',
                        offset: true,
                        position: 'left',
                        labels: ['Yes', 'No'],
                        title: {
                            display: true,
                            text: 'Availability',
                        },
                        ticks: {
                          beginAtZero: true,
                        },
                        border: {
                            color: 'rgba(0, 255, 0, 0.75)',
                        }
                    } : {
                        min: window.webcheckups_use_optimized_scale ?
                            (window.webcheckups_optimized_min || minValue) : minValue,
                        max: window.webcheckups_use_optimized_scale ?
                            (window.webcheckups_optimized_max || maxValue) : maxValue,
                        title: {
                            display: true,
                            text: 'Response Time (in seconds)',
                        },
                    }
                },
                hover: {
                    mode: 'nearest',
                    intersect: true
                },
                interaction: {
                    mode: 'x',
                    intersect: true,
                },
                plugins: {
                    title: {
                        display: true,
                        text: title,
                        font: {
                            size: 15,
                        },
                        padding: {
                            bottom: 10,
                        },
                    },
                    tooltip: {
                        enabled: true,
                        titleFontSize: 20,
                        bodyFontSize: 19,
                        titleAlign: 'center',
                        position: 'nearest',
//                        external: externalTooltipHandler,
                        callbacks: {
                            title: function(tooltipItems) {
                                let res = '';
                                tooltipItems.forEach(function(tooltipItem) {
                                    if (tooltipItem.raw?.x) {
                                        res = makeTitle(tooltipItem.raw, variant);
                                    }
                                });
                                return res;
                            },
                            label: function(tooltipItem)
                            {
                                const name = upperFirst(service);
                                if (tooltipItem.raw?.max) {
                                    const res = [' ' + name + ' Checkup Details: ', ''];
                                    res.push(' Total Checkups: ' + formatNumbers(tooltipItem.raw?.seq, 0));
                                    res.push(' Fastest Response: ' + formatNumbers(tooltipItem.raw?.min) + ' sec ');
                                    res.push(' Slowest Response: ' + formatNumbers(tooltipItem.raw?.max) + ' sec ');
                                    res.push(' Average Response Time: ' + formatNumbers(tooltipItem.raw?.y) + ' sec ');
                                    return res;
                                } else if (tooltipItem.raw?.on || tooltipItem.raw?.off) {
                                    const res = [' ' + name +' Availability: ', ''];
                                    res.push(' Total Checkups: ' + formatNumbers(tooltipItem.raw?.cnt, 0));
                                    res.push(' Successful Checkups: ' + formatNumbers(tooltipItem.raw?.on, 0));
                                    res.push(' Failed Checkups: ' + formatNumbers(tooltipItem.raw?.off, 0));
                                    return res;
                                } else {
                                    return isNaN(tooltipItem.raw?.y) ?
                                        ' ' + name + ' Status: ' + tooltipItem.raw?.y :
                                        ' ' + name + ' Response Time: ' + formatNumbers(tooltipItem.raw?.y) + ' sec';
                                }
                            },
                        }
                    },
                }
            }
        }
    );
}

/**
 * Formats date/time for chart tooltips
 *
 * @param {Object} entry - Data point with x property (timestamp)
 * @param {string} variant - Data granularity type
 * @returns {string} Formatted date/time string
 *
 * @description
 * Formats timestamps based on granularity:
 * - 'daily'/'errDaily': Date only
 * - 'hourly': Date + time range (hour ±30 min)
 * - 'errHourly': Date + failure time range
 * - 'realtime': Date + exact time
 */
function makeTitle(entry, variant) {
    const d = new Date(entry.x);
    let res = d.toLocaleDateString(localLang);

    if (variant == 'daily' || variant == 'errDaily') {
        return res;
    } else if (variant == 'hourly') {
        d.setHours(d.getHours()-1);
        d.setMinutes(d.getMinutes() + 30);
        res +=  ' ' + printHoursMinutes(d);
        d.setHours(d.getHours()+1);
        d.setMinutes(d.getMinutes() -1);
        res +=  ' - ' + printHoursMinutes(d);
        return res;
    } else if (variant == 'errHourly') {
        d.setHours(d.getHours()-1);
        d.setMinutes( 30);
        const d1 = new Date(entry.from).getTime() >= d.getTime() ? new Date(entry.from) : d;
        res +=  ' ' + printHoursMinutes(d1);
        d.setHours(d.getHours()+1);
        d.setMinutes(d.getMinutes() -1);
        if (entry.from != entry.until) {
            const d2 = new Date(entry.until).getTime() < d.getTime() ? new Date(entry.until) : d;
            res +=  ' - ' + printHoursMinutes(d2);
        }
        return res;
    }
    else {
        res += ' ' + printHoursMinutes(d);
    }
    return res;
}

/**
 * Updates granularity button states (DEPRECATED - replaced by buildChartNavigation)
 *
 * @deprecated Use buildChartNavigation instead
 * @param {Array<string>} availableGranularities - Available granularity options
 * @param {string} currentVariant - Currently active variant
 * @param {boolean} [isYesNo=false] - Whether in YesNo mode
 * @returns {void}
 *
 * @description
 * Legacy function for granularity controls.
 * Now handled by buildChartNavigation.
 */
function updateGranularityControls(availableGranularities, currentVariant, isYesNo = false) {
    const wrapper = jQuery('#granularity_controls_wrapper');
    const buttons = wrapper.find('.granularity-btn');
    const infoSpan = jQuery('#granularity_info');

    // Verstecken für YesNo Charts (Doughnut, Failed Charts etc.)
    if (isYesNo) {
        wrapper.hide();
        return;
    }

    // Verstecken wenn nur eine Granularity verfügbar
    if (availableGranularities.length <= 1) {
        wrapper.hide();
        return;
    }

    // Wrapper anzeigen
    wrapper.show();

    // Buttons aktivieren/deaktivieren basierend auf verfügbaren Daten
    buttons.each(function() {
        const granularity = jQuery(this).data('granularity');
        const isAvailable = availableGranularities.includes(granularity);
        const isActive = granularity === currentVariant;

        jQuery(this)
            .prop('disabled', !isAvailable)
            .toggleClass('active', isActive);
    });

    // Info-Text anzeigen
    const granularityLabels = {
        'daily': 'Showing daily aggregated data',
        'hourly': 'Showing hourly aggregated data',
        'realtime': 'Showing all individual checkup results'
    };
    infoSpan.text(granularityLabels[currentVariant] || '');
}

/**
 * Global vars for current chart data (for refresh)
 */
window.webcheckups_current_chart_data = null;
window.webcheckups_current_element_id = null;
window.webcheckups_current_show_all = true;

/**
 * Loads and displays statistics charts for a checkup
 *
 * @async
 * @param {number} checkupId - Checkup ID to load
 * @param {string} timespan - Time period ('day', 'week', 'month', etc.)
 * @param {string} elementId - Container element ID
 * @param {boolean} [showAll=true] - Show all chart types
 * @param {string|null} [granularityOverride=null] - Manual granularity override
 * @param {boolean} [disableGranularityControls=false] - If true, hides granularity controls completely
 * @param {boolean} [hideChartTypeSelector=false] - If true, hides chart type dropdown (dashboard slider)
 * @returns {Promise<void>}
 *
 * @description
 * Main entry point for chart rendering.
 *
 * **Process:**
 * 1. Calculates fromDate based on timespan
 * 2. Calls manage_api_data with caching support
 * 3. Stores data globally for refresh functionality
 * 4. Calls drawStatGraphic to render charts
 *
 * **Global variables set:**
 * - webcheckups_current_chart_data: For refresh
 * - webcheckups_current_element_id: Container ID
 * - webcheckups_current_show_all: Chart visibility flag
 *
 * **Usage:**
 * - Dashboard slider: disableGranularityControls=true (small charts don't need controls)
 * - Checkup details page: disableGranularityControls=false (full controls visible)
 *
 * Uses manage_api_data for automatic caching (dynamic TTL from API response).
 */
async function showGraphic(checkupId, timespan, elementId, showAll=true, granularityOverride=null, disableGranularityControls=false, hideChartTypeSelector=false) {
    const range = getTimespanDateRange(timespan);
    const fromDate = range.startDate.toISOString().substring(0, 14) + '00';
    const apiEndpoint = 'statistics?filter[checkupId]=' + checkupId +
        (fromDate ? '&filter[fromDate]=' + fromDate : '');

    try {
        const data = await manage_api_data(apiEndpoint, 'get');

        if (data?.errors) {
            console.error('Error loading statistics:', data.errors);
            return;
        }

        window.webcheckups_current_chart_data = data;
        window.webcheckups_current_element_id = elementId;
        window.webcheckups_current_show_all = showAll;

        await drawStatGraphic(data, elementId, showAll, granularityOverride, disableGranularityControls, hideChartTypeSelector);
    } catch (error) {
        console.error('Error in showGraphic:', error);
    }
}

function setupChartNavigationHandlers() {
    const container = jQuery('#details_chart');

    // Chart Type Dropdown Handler (Event Delegation)
    container.off('change', '#dropdown_chart_type').on('change', '#dropdown_chart_type', function() {
        const selectedType = jQuery(this).val();

        if (selectedType === 'failed') {
            if (typeof drawFailedGraphic === 'function') {
                drawFailedGraphic(
                    window.webcheckups_current_chart_data,
                    'details_chart',
                    true
                );
            }
        } else {
            const serviceData = window.webcheckups_current_chart_data?.data?.find(
                d => d.type?.toLowerCase() === selectedType
            );

            if (serviceData) {
                createSuccessGraphic(
                    'details_chart',
                    [serviceData],
                    [],
                    window.webcheckups_current_granularity_override || null
                );
            }
        }
    });

    // Granularity Buttons Handler (Event Delegation)
    container.off('click', '.granularity-btn').on('click', '.granularity-btn', function(e) {
        e.preventDefault();

        if (jQuery(this).prop('disabled')) return;

        const selectedGranularity = jQuery(this).data('granularity');
        const isAlreadyActive = jQuery(this).hasClass('active');
        const granularityOverride = isAlreadyActive ? null : selectedGranularity;

        // Override speichern
        window.webcheckups_current_granularity_override = granularityOverride;

        // Chart neu rendern
        if (window.webcheckups_current_chart_data) {
            (async () => {
                await drawStatGraphic(
                    window.webcheckups_current_chart_data,
                    'details_chart',
                    true,
                    granularityOverride
                );
            })();
        } else {
            const currentCheckupId = parseInt(getElById('dropdown_checkup_list').value.trim());
            const currentPeriod = getElById('dropdown_checkup_result_period').value.trim();
            showGraphic(currentCheckupId, currentPeriod, 'details_chart', true, granularityOverride);
        }
    });
}

jQuery(document).ready(function () {
    jQuery(document).on('change', 'select#chartNav', function (event) {
        jQuery(this).parents('div.graphics').find('canvas:visible').hide();
        jQuery(this).parents('div.graphics').find('canvas.' + this.value).show();
    });
});
