/* /lib/js/statistics.js */
/*!
 * statistics Functions for Website Checkups Plugin v1.2.0
 * https://website-checkups.com
 * Extracted from admin.js and charts.js during refactoring
 * Released under the GPLv2 license
 */

/**
 * Aggregates checkup data by date
 *
 * @async
 * @param {Array<Object>} checkups - Array of checkup objects to aggregate
 * @param {string} timespan - Time period to aggregate (e.g., 'week', 'month')
 * @returns {Promise<Object>} Object with dates as keys and {successful, failed} counts as values
 *
 * @description
 * Loads statistics for all checkups in parallel and aggregates by date.
 * Uses ONLY existing Charts.js functions (getIsYesNo, prepareOkData, prepareFailData).
 *
 * Process:
 * 1. Calculates date range from timespan
 * 2. Fetches statistics for all checkups in parallel
 * 3. Processes data using Charts.js functions
 * 4. Aggregates successful and failed checks by date
 *
 * @example
 * const data = await aggregateCheckupsByDate(checkups, 'month');
 * // Returns: {'2024-01-15': {successful: 100, failed: 2}, ...}
 */
async function aggregateCheckupsByDate(checkups, timespan) {
    const dataByDate = {};
    const isDashboard = checkups.length > 1; // Dashboard shows multiple services

    // Calculate fromDate from timespan
    // Timeline cache will handle checking if data exists and fetching gaps
    const range = getTimespanDateRange(timespan);
    const startDate = new Date(range.startDate);
    startDate.setMinutes(0, 0, 0);
    const fromDate = startDate.toISOString().substring(0, 14) + '00';

    // console.log('📊 Loading calendar data for', checkups.length, 'checkups, timespan:', timespan, 'fromDate:', fromDate);

    // Load statistics for all checkups in parallel
    const promises = checkups.map(async checkup => {
        const checkupId = checkup.checkupId.toString();
        const url = `statistics?filter[checkupId]=${checkupId}&filter[fromDate]=${fromDate}`;

        try {
            const response = await manage_api_data(url, 'get', { variantNeeded: 'daily' });

            if (response?.data) {
                const isYesNo = getIsYesNo(response);
                const variantData = {
                    variant: 'daily',
                    isYesNo: isYesNo,
                    count: 0
                };

                const okData = prepareOkData(response, variantData);
                const failData = prepareFailData(response);

                const okCount = countSequencies(okData);
                const failCount = countSequencies(failData, false);
                // console.log(`  Checkup ${checkupId} (${checkup.name}):`, okData.length, 'days,', okCount, 'OK checks,', failCount, 'fail checks');

                return { checkup, okData, failData, isYesNo, success: true };
            }
            console.warn(`  No data for checkup ${checkupId}`);
            return { checkup, okData: [], failData: [], success: false };
        } catch (err) {
            console.warn(`Failed to load statistics for checkup ${checkupId}:`, err);
            return { checkup, okData: [], failData: [], success: false };
        }
    });

    const results = await Promise.all(promises);

    // Aggregate data by date
    results.forEach(result => {
        if (!result.success) return;

        // Track per-service stats for dashboard tooltips
        const serviceStats = {}; // dateStr -> {successful, failed, total}

        // Process OK data
        result.okData.forEach(point => {
            const dateStr = point.x.split('T')[0];

            if (!dataByDate[dateStr]) {
                dataByDate[dateStr] = {
                    successful: 0,
                    failed: 0,
                    services: isDashboard ? [] : null
                };
            }

            if (!serviceStats[dateStr]) {
                serviceStats[dateStr] = { successful: 0, failed: 0 };
            }

            // seq (normal checks) or cnt (YesNo checks)
            const count = point.seq || point.cnt || 1;
            dataByDate[dateStr].successful += count;
            serviceStats[dateStr].successful += count;

            // Collect response time stats if available (numeric mode)
            if (!isDashboard && point.min !== undefined && point.max !== undefined) {
                if (!dataByDate[dateStr].responseTimes) {
                    dataByDate[dateStr].responseTimes = [];
                    dataByDate[dateStr].hasResponseTimes = true;
                }
                // Collect min/max/avg for this data point
                dataByDate[dateStr].responseTimes.push({
                    min: point.min,
                    max: point.max,
                    avg: point.y,
                    count: point.seq || 1
                });
            }
        });

        // Process fail data
        result.failData.forEach(point => {
            const dateStr = point.performed.split('T')[0];

            if (!dataByDate[dateStr]) {
                dataByDate[dateStr] = {
                    successful: 0,
                    failed: 0,
                    services: isDashboard ? [] : null
                };
            }
            if (!serviceStats[dateStr]) {
                serviceStats[dateStr] = { successful: 0, failed: 0 };
            }
            dataByDate[dateStr].failed += point.seq || 1;
            serviceStats[dateStr].failed += point.seq || 1;
        });

        // Add service to per-date service list (for dashboard tooltips)
        if (isDashboard) {
            Object.keys(serviceStats).forEach(dateStr => {
                const stats = serviceStats[dateStr];
                if (stats.successful > 0 || stats.failed > 0) {
                    dataByDate[dateStr].services.push({
                        name: result.checkup.name,
                        successful: stats.successful,
                        failed: stats.failed,
                        total: stats.successful + stats.failed
                    });
                }
            });
        }
    });
    // console.log(`    Added ${failure.count} failures to ${dateStr}`);

    // Calculate aggregated min/max/avg for each day (single service only)
    if (!isDashboard) {
        Object.keys(dataByDate).forEach(dateStr => {
            const day = dataByDate[dateStr];
            if (day.responseTimes && day.responseTimes.length > 0) {
                // Find overall min/max across all data points for this day
                day.min = Math.min(...day.responseTimes.map(rt => rt.min));
                day.max = Math.max(...day.responseTimes.map(rt => rt.max));

                // Calculate weighted average based on count
                const totalCount = day.responseTimes.reduce((sum, rt) => sum + rt.count, 0);
                const weightedSum = day.responseTimes.reduce((sum, rt) => sum + (rt.avg * rt.count), 0);
                day.avg = weightedSum / totalCount;

                // Clean up temporary array
                delete day.responseTimes;
            }
        });
    }

    // console.log('🔧 DEBUG aggregateCheckupsByDate: Collected data for dates:', Object.keys(dataByDate).sort());
    return dataByDate;
}

/**
 * Calculates start and end date for given timespan
 *
 * @param {string} timespan - Time period identifier
 * @returns {Object} Object with startDate and endDate properties (Date objects)
 *
 * @description
 * Supported timespans:
 * - 'day', '3days', 'week', '2weeks'
 * - 'month', '2month', '3month', '6month', '12month'
 * - '2monthes', '3monthes', '6monthes', 'year', '18monthes' (legacy variants)
 *
 * Start date: midnight (00:00:00.000) of calculated date
 * End date: end of today (23:59:59.999)
 *
 * Default: 7 days (week) if timespan not recognized
 */
function getTimespanDateRange(timespan) {
    const endDate = new Date();
    endDate.setHours(23, 59, 59, 999);

    // Keep current time, just subtract days
    const startDate = new Date();
//    startDate.setHours(0, 0, 0, 0);

    switch (timespan) {
        case 'day':
            startDate.setDate(startDate.getDate() - 1);
            break;
        case '3days':
            startDate.setDate(startDate.getDate() - 3);
            break;
        case 'week':
            startDate.setDate(startDate.getDate() - 7);
            break;
        case '2weeks':
            startDate.setDate(startDate.getDate() - 14);
            break;
        case 'month':
            startDate.setMonth(startDate.getMonth() - 1);
            break;
        case '2month':
        case '2monthes':
            startDate.setMonth(startDate.getMonth() - 2);
            break;
        case '3month':
        case '3monthes':
            startDate.setMonth(startDate.getMonth() - 3);
            break;
        case '6month':
        case '6monthes':
            startDate.setMonth(startDate.getMonth() - 6);
            break;
        case '12month':
        case 'year':
            startDate.setMonth(startDate.getMonth() - 12);
            break;
        case '18month':
        case '18monthes':
            startDate.setMonth(startDate.getMonth() - 18);
            break;            
        default:
            startDate.setDate(startDate.getDate() - 7);
    }

    return { startDate, endDate };
}

/**
 * Prepares successful checkup data for chart rendering
 *
 * @param {Object} data - Raw API response data
 * @param {Object} variantData - Variant configuration object
 * @param {string} variantData.variant - Granularity: 'daily', 'hourly', or 'realtime'
 * @param {boolean} variantData.isYesNo - Whether data is binary (Yes/No) or numeric
 * @returns {Array<Object>} Array of formatted data points for charts
 *
 * @description
 * Transforms API response into chart-ready format based on granularity:
 *
 * **For YesNo data (binary availability):**
 * - daily/hourly: {x: timestamp, y: 'Yes'|'No', on: count, off: count, cnt: total}
 * - realtime: {x: timestamp, y: 1} for each individual check
 *
 * **For numeric data (response times):**
 * - daily/hourly: {x: timestamp, y: avg, min: min, max: max, seq: count}
 * - realtime: {x: timestamp, y: responseTime} for each individual check
 *
 * Special handling: For YesNo realtime, also includes failed checks as 'No' entries
 */
function prepareOkData(data, variantData) {
    let res = [];
    const isYesNo = variantData.isYesNo;
    const variant = variantData.variant;
    const checkData = data?.data.filter(i => i.status == 1);

    // Sort segments by 'from' timestamp to ensure chronological processing
    // Prevents unsorted data when multiple segments are processed
    checkData.sort((a, b) => {
        const dateA = new Date(a.from);
        const dateB = new Date(b.from);
        return dateA - dateB;
    });
    
    for (let i = 0; i < checkData.length; i++)
    {
        const tmpData = checkData[i];
        if (isYesNo) {
            const data1 = variant == 'daily' ? tmpData?.data?.daily : tmpData?.data?.hourly ?? {};
            if (variant !== 'realtime') {
                for (let t in data1) {
                    const data2 = data1[t];
                    res.push({
                        x: variant == 'daily' ? setMidday(t) : getCuttedDateString(t.replace(' ', 'T') + 'Z'),
                        y: data2.off > 0 ? 'No' : 'Yes',
                        on: data2.on,
                        off: data2.off,
                        cnt: data2.cnt,
                    });
                }
            } else {
                let timeObj = new Date(tmpData.from);
                const intervals = tmpData?.data?.intervals ?? [];
                for (let j = 0; j < tmpData.counter; j++) {
                    const mewInterval = intervals[j.toString()] ?? tmpData.interval ?? tmpData.timeInterval;
                    res.push({
                        x: getCuttedDateString(timeObj),
                        y: 1,
                    });
                    timeObj.setMinutes(timeObj.getMinutes() + mewInterval);
                }
            }
        } else {
            const data1 = variant == 'daily' ? tmpData?.data?.daily :
                 variant == 'hourly' ? tmpData?.data?.hourly :
                 (tmpData?.data?.time || tmpData?.data?.hourly);// Fallback to hourly if time missing

            if (variant !== 'realtime') {
                for (let t in data1) {
                    const data2 = data1[t];
                    res.push({
                        x: variant == 'daily' ? setMidday(t) : getCuttedDateString(t.replace(' ', 'T') + 'Z'),
                        y: data2.avg,
                        min: data2.min,
                        max: data2.max,
                        seq: data2.cnt,
                    });
                }
            } else if (data1?.length) { // time array exists
                let timeObj = new Date(tmpData.from);
                const intervals = tmpData?.data?.intervals ?? [];
                 // console.log('🔍 Realtime data:', {
                 //     from: tmpData.from,
                 //     firstTimeObj: timeObj.toISOString(),
                 //     dataLength: data1.length,
                 //     firstInterval: intervals['0'] ?? tmpData.interval
                 // });

                for (let j = 0; j < data1.length; j++) {
                    const mewInterval = intervals[j.toString()] ?? tmpData.interval;
                    const cuttedDate = getCuttedDateString(timeObj);
                    // if (j < 3 || j > data1.length - 3) {
                    //     console.log(`  Entry ${j}: time=${cuttedDate}, y=${data1[j]}`);
                    // }
                    res.push({
                        x: cuttedDate,
                        y: data1[j],
                    });
                    timeObj.setMinutes(timeObj.getMinutes() + mewInterval);
                }
            } else if (tmpData?.data?.hourly) {  // Fallback: use hourly when time doesn't exist
                // Process hourly data like 'hourly' variant would
                for (let t in tmpData.data.hourly) {
                    const data2 = tmpData.data.hourly[t];
                    res.push({
                        x: getCuttedDateString(t.replace(' ', 'T') + 'Z'),
                        y: data2.avg,
                        min: data2.min,
                        max: data2.max,
                        seq: data2.cnt,
                    });
                }
            }
        }
    }
    // add failed entries to ok data
    if (isYesNo && variant === 'realtime') {
        const checkData2 = data?.data.filter(i => i.status == 0);
        for (let i = 0; i < checkData2.length; i++) {
            const tmpData = checkData2[i];
            let timeObj = new Date(tmpData.from);
            const intervals = tmpData?.data?.intervals ?? [];
            for (let j = 0; j < tmpData.counter; j++) {
                const mewInterval = intervals[j.toString()] ?? tmpData.interval ?? tmpData.timeInterval;
                res.push({
                    x: getCuttedDateString(timeObj),
                    y: 'No',
                });
                timeObj.setMinutes(timeObj.getMinutes() + mewInterval);
            }
        }
    }
    // console.log('res!!ok', res);
    // console.log('🔍 prepareOkData BEFORE sort:', {
    //     variant: variantData?.variant,
    //     isYesNo: variantData?.isYesNo,
    //     resLength: res.length,
    //     first3: res.slice(0, 3).map(r => ({x: r.x, y: r.y})),
    //     last3: res.slice(-3).map(r => ({x: r.x, y: r.y}))
    // });
    // Sort result chronologically by x (date/time)
    // This ensures consistent order regardless of data source
    res.sort((a, b) => {
        if (!a.x || !b.x) {
            console.error('❌ Sort error: missing x value', {a, b});
            return 0;
        }
        const dateA = new Date(a.x);
        const dateB = new Date(b.x);
        if (isNaN(dateA.getTime()) || isNaN(dateB.getTime())) {
            console.error('❌ Sort error: invalid date', {
                a_x: a.x,
                b_x: b.x,
                dateA: dateA.toString(),
                dateB: dateB.toString()
            });
            return 0;
        }
        return dateA - dateB;
    });

    // console.log('✅ prepareOkData AFTER sort:', {
    //     resLength: res.length,
    //     first3: res.slice(0, 3).map(r => ({x: r.x, y: r.y})),
    //     last3: res.slice(-3).map(r => ({x: r.x, y: r.y})),
    //     firstDate: res[0]?.x,
    //     lastDate: res[res.length - 1]?.x
    // });

    return res;
}

/**
 * Prepares failed checkup data for chart rendering
 *
 * @param {Object} data - Raw API response data
 * @returns {Array<Object>} Array of failure records
 *
 * @description
 * Extracts and formats failure information from API response.
 *
 * **Output format:**
 * {
 *   performed: timestamp,
 *   error: errorMessage,
 *   seq: consecutiveFailureCount,
 *   until: lastFailureTimestamp
 * }
 *
 * **Consecutive error grouping:**
 * Groups consecutive failures with the same error message into single records
 * to reduce data volume and improve chart readability.
 */
function prepareFailData(data) {
    let res = [];
    const checkData = data?.data.filter(i => i.status == 0);

    // Sort segments by 'from' timestamp to ensure chronological processing
    checkData.sort((a, b) => {
        const dateA = new Date(a.from);
        const dateB = new Date(b.from);
        return dateA - dateB;
    });

    for (let i = 0; i < checkData.length; i++)
    {
        const tmpData = checkData[i];
        if (!tmpData.data) {
            res.push({
                performed: getCuttedDateString(tmpData.from),
                error: tmpData.description,
                seq: tmpData.counter,
                until: getCuttedDateString(tmpData.until),
            });
        } else {
            let timeObj = new Date(tmpData.from);
            const intervals = tmpData?.data?.intervals ?? [];
            const errorRefs = tmpData?.data?.errReferences ?? [];
            const errorTexts = tmpData?.data?.errMessages ?? [];
            let lastError = {};

            for (let j = 0; j < tmpData.counter; j++) {
                const mewInterval = intervals[j.toString()] ?? tmpData.interval ?? tmpData.timeInterval;
                const newRef = errorRefs[j.toString()] ?? -1;
                const newErrorDesc = newRef >= 0 ? errorTexts[ newRef.toString() ] : tmpData.description;
                if (lastError?.error == newErrorDesc) {
                    lastError.seq++;
                    lastError.until = getCuttedDateString(timeObj);
                } else {
                    if (lastError?.error) {
                        res.push(lastError);
                    }
                    lastError = {
                        performed: getCuttedDateString(timeObj),
                        error: newErrorDesc,
                        seq: 1,
                        until: getCuttedDateString(timeObj),
                    }
                }
                timeObj.setMinutes(timeObj.getMinutes() + mewInterval);
            }
            if (lastError?.error) {
                res.push(lastError);
            }
        }
    }
//    console.log('res!!fail', res);

    // Sort result chronologically by x (date/time)
    // This ensures consistent order regardless of data source
    res.sort((a, b) => {
        const dateA = new Date(a.performed);
        const dateB = new Date(b.performed);
        return dateA - dateB;
    });

    return res;
}

/**
 * Counts total checkup sequences from data array
 *
 * @param {Array<Object>} data - Array of checkup data points
 * @param {boolean} [countAll=true] - If true, counts all checks; if false, only counts 'off' checks
 * @returns {number} Total count of checkup sequences
 *
 * @description
 * Sums up checkup counts from data points.
 * Handles different data formats: seq (sequence count), cnt (total count), or defaults to 1.
 * For YesNo data with countAll=false, only counts failures (off count).
 */
function countSequencies(data, countAll=true) {
    let res = 0;
    for (let i = 0; i < data.length; i++) {
        res += data[i].seq ?? (countAll ? data[i].cnt : data[i].off) ?? 1;
    }
    return res;
}

/**
 * Determines if checkup data is binary (Yes/No) or numeric (response times)
 *
 * @param {Object} data - Raw API response data
 * @returns {boolean} True if data is binary (availability only), false if numeric (response times)
 *
 * @description
 * Checks successful checkup records for presence of numeric data (avg, time).
 * If ANY record contains response time data, returns false (numeric mode).
 * If ALL records contain only binary status, returns true (YesNo mode).
 *
 * YesNo mode: Charts show only "Yes" or "No" status
 * Numeric mode: Charts show actual response times with min/max/avg
 */
function getIsYesNo(data){
    let res = 0;
    const checkData = data?.data.filter(i => i.status == 1);
    for (let i = 0; i < checkData.length; i++) {
        const tmpData = checkData[i];
        const hourly = tmpData?.data?.hourly ?? {};
        const firstKeyHourly = Object.keys(hourly)[0] ?? null;
        const daily = tmpData?.data?.daily ?? {};
        const firstKeyDaily = Object.keys(daily)[0] ?? null;

        if (tmpData?.data?.time ||
            (firstKeyHourly && hourly[firstKeyHourly]?.avg) ||
            (firstKeyDaily && daily[firstKeyDaily]?.avg))
        {
            return false;
        } else {
            res++;
        }
    }
    return res > 0;
}

/**
 * Determines optimal data granularity for chart display
 *
 * @param {Object} statData - Raw statistics data from API
 * @param {string|null} [granularityOverride=null] - Manual granularity override ('daily', 'hourly', 'realtime')
 * @returns {Object} Variant configuration: {variant, isYesNo, count, availableGranularities, isManualOverride}
 *
 * @description
 * Automatically selects best granularity based on data volume and check frequency.
 *
 * **Thresholds (adaptive based on isYesNo):**
 * - Daily: >= 8 (YesNo) or 21 (numeric) data points
 * - Hourly: > 56 (YesNo) or 168 (numeric) points AND > 24 checks/day
 * - Realtime: < 150 (YesNo) or 900 (numeric) points AND < 3.5x checks/day
 *
 * **Priority:** granularityOverride > automatic calculation
 *
 * Returns available granularities for UI controls.
 */
function getVariant(statData, granularityOverride = null) {
    const isYesNo = getIsYesNo(statData);
    const dayLimit = isYesNo ? 8 : 21;
    const hourlyLimit = isYesNo ? 56 : 168;
    const allLimit = isYesNo ? 150 : 900;

    let dailyCnt = 0;
    let hourlyCnt = 0;
    let allCnt = 0;
    const maxPerDay = 1440 / (statData?.data.reduce((res, obj) => {
        if (res === null || obj.interval < res) return obj.interval;
        return res;
    }, null) ?? 60);

    // Count available data per granularity
    for (let i = 0; i < statData?.data?.length; i++) {
        const tmpData = statData?.data[i];
        dailyCnt += tmpData?.data?.daily ? Object.keys(tmpData.data.daily).length : 0;
        hourlyCnt += tmpData?.data?.hourly ? Object.keys(tmpData.data.hourly).length : 0;
        allCnt += tmpData?.data?.time ? tmpData?.data?.time.length : 0;
    }

    // Determine available granularities
    const availableGranularities = [];
    if (dailyCnt > 0) availableGranularities.push('daily');
    if (hourlyCnt > 0) availableGranularities.push('hourly');
    if (allCnt > 0) availableGranularities.push('realtime');

    // If override set AND available, use it
    if (granularityOverride && availableGranularities.includes(granularityOverride)) {
        return {
            variant: granularityOverride,
            isYesNo: isYesNo,
            count: granularityOverride === 'realtime' ? allCnt :
                granularityOverride === 'hourly' ? hourlyCnt : dailyCnt,
            availableGranularities: availableGranularities,
            isManualOverride: true
        };
    }

    // Automatic determination (original logic)
    const currentVariant = dailyCnt >= dayLimit ? 'daily' :
        hourlyCnt > hourlyLimit && maxPerDay > 24 ? 'hourly' :
            allCnt > 0 && allCnt < Math.min(allLimit, maxPerDay * 3.5) ? 'realtime' : 'hourly';

    return {
        variant: currentVariant,
        isYesNo: isYesNo,
        count: currentVariant === 'realtime' ? allCnt :
            currentVariant === 'hourly' ? hourlyCnt : dailyCnt,
        availableGranularities: availableGranularities,
        isManualOverride: false
    };
}

/**
 * Calculates min/max values and statistics for chart scaling
 *
 * @param {Array<Object>} data - Array of successful checkup data points
 * @param {Array<Object>} failed - Array of failed checkup data points
 * @returns {Object} Statistics object with min, max, avg, uptimePercent, and optimized scaling
 *
 * @description
 * Calculates comprehensive statistics for chart rendering:
 *
 * **Basic statistics:**
 * - min/max: Minimum and maximum response times
 * - avg: Average response time
 * - uptimePercent: Uptime percentage (successful / total)
 * - total/failed: Count of successful and failed checks
 *
 * **Smart outlier handling:**
 * - Removes outliers beyond 91-96th percentile (adaptive)
 * - Groups response times into 5ms buckets for distribution analysis
 * - Detects significant gaps (2x-20x jumps) to cut off outliers
 *
 * **Optimized Y-axis scaling:**
 * - optimizedMin/Max: 95th percentile-based scaling for better readability
 * - showsAllData: Flag indicating if all data points are shown
 * - Adds 10% padding for visual clarity
 *
 * **Adaptive rounding:**
 * - > 5 sec: rounds to 0.1 sec
 * - < 0.2 sec: rounds to 0.01 sec
 * - Otherwise: rounds to 0.01 sec
 */
function getMinMaxValues(data, failed) {
    const totalSum = countSequencies(data);
    const failedSum = countSequencies(failed, false);
    const uptimePercent = totalSum > 0 ? Math.round(totalSum / (totalSum + failedSum)*10000)/100 : 0;
    const percent = totalSum > 1000 ? 96 : totalSum > 500 ? 95 : totalSum > 250 ? 94 : totalSum > 100 ? 92 : 91;

    const okLimit = Math.ceil(totalSum/100 * percent);
    const minLimit =  Math.ceil(totalSum/100 * 75);
    let roundedTime = 0;
    let sumTime = 0;
    let distribution = [];
    let result = {
        min: 9999,
        max: 0,
        uptimePercent: uptimePercent,
        total: totalSum,
        failed: failedSum,
        optimizedMin: 0,
        optimizedMax: 0,
        showsAllData: true
    };
    const okValues = [];
    for (let i = 0; i < data.length; i++) {
        sumTime += data[i].y;
        okValues.push(data[i].y);
        roundedTime = Math.round(data[i].y * 200) / 200;
//      console.log('roundedTime-' + i, roundedTime);
        if (roundedTime < 0.005) {
            roundedTime = 0.005;
        }

        if (!distribution[roundedTime]) {
            distribution[roundedTime] = 1;
        } else {
            distribution[roundedTime]++;
        }
    }

    //    console.log('distribution', distribution);
    if (totalSum > 1) {
        result.avg = Math.round(sumTime / data.length * 1000) / 1000;
    }
    const distributionKeys = Object.keys(distribution).sort();
    let currentEntries = 0;
    result.allData = true;
    for (let i in distributionKeys) {
        const time = distributionKeys[i];
        const floatTime = parseFloat(time);
        result.min = Math.min(result.min, floatTime);
        result.max = Math.max(result.max, floatTime);
        currentEntries += distribution[time];
        if (currentEntries > okLimit) {
            result.break = percent + '%';
//            result.currentEntries = currentEntries;
//            result.okLimit = okLimit;
//            result.totalSum = totalSum;
            result.allData = (currentEntries == totalSum);
            break;
        } else if (currentEntries > minLimit &&
            ((floatTime > result.min * 2 && result.min >= 0.5) ||
                (floatTime > result.min * 2.5 && result.min >= 0.25) ||
                (floatTime > result.min * 3 && result.min >= 0.15) ||
                (floatTime > result.min * 4 && result.min >= 0.1) ||
                (floatTime > result.min * 7 && result.min >= 0.05) ||
                (floatTime > result.min * 10 && result.min >= 0.01) ||
                (floatTime > result.min * 20 && result.min >= 0.001))
        ) {
            result.break = floatTime + ' sec';
            result.allData = false;
            break;
        }
    }
    if (!result.allData) {
        const HighestRoundedTime = distributionKeys[distributionKeys.length -1];
        result.allData = (result.max > 5 && Math.ceil(result.max * 10) / 10 >= HighestRoundedTime) ||
            (result.max < 0.2 && Math.ceil(result.max * 1.15 * 100) / 100 >= HighestRoundedTime) ||
            (result.max <= 5 && result.max >= 0.2 && Math.ceil(result.max * 1.05 * 100) / 100 >= HighestRoundedTime);
    }

//    console.log('getMinMaxTmp', result);
    if (result.max > 5) {
        result.max = result.allData ? Math.ceil(result.max * 10) / 10 : Math.ceil(result.max * 2) / 2;
        result.min = Math.floor(result.min * 0.975 * 20) / 20;
    } else if (result.max < 0.2) {
        result.max = result.allData ? Math.ceil(result.max * 1.15 * 100) / 100 : Math.ceil(result.max * 1.25 * 10) / 10;
        result.min = Math.floor(result.min * 0.85 * 20) / 20;
    } else {
        result.max = result.allData ? Math.ceil(result.max * 1.05 * 100) / 100 : Math.ceil(result.max * 1.15 * 10) / 10;
        result.min = Math.floor(result.min * 0.95 * 20) / 20;
    }

    if (result.min === result.max) {
        result.max += 0.02;
    }

    // Calculate optimized Y-axis scaling for better readability
    if (okValues.length > 0) {
        const sortedValues = okValues.slice().sort((a, b) => a - b);

        // 95th percentile for optimized max value
        const percentileIndex = Math.floor(sortedValues.length * 0.95);
        const percentile95 = sortedValues[percentileIndex - 1] || sortedValues[sortedValues.length - 1];

        result.optimizedMin = Math.max(0, sortedValues[0] * 0.9);
        result.optimizedMax = percentile95 * 1.1;
        result.showsAllData = percentile95 === sortedValues[sortedValues.length - 1];
    } else {
        result.optimizedMin = result.min;
        result.optimizedMax = result.max;
        result.showsAllData = true;
    }
//    console.log('getMinMax', result);
    return result;
}

/**
 * Aggregates and prepares failed checkup data for failure chart
 *
 * @param {Array<Object>} graphDataFail - Array of individual failure records
 * @returns {Object} Aggregated failure data: {max, total, unit, graphData[]}
 *
 * @description
 * Groups failures by time interval (minute/hour or day) for cleaner visualization.
 *
 * **Grouping logic:**
 * - Short timespan (< 3 hours): Group by rounded minute/hour
 * - Long timespan: Group by day
 *
 * **For each time bucket:**
 * - count: Total number of failures
 * - from/until: First and last failure timestamp
 * - error: Most frequent error message (by occurrence count)
 * - errSeq: Sequence count of most frequent error
 *
 * **Output structure:**
 * {
 *   max: highest failure count in any bucket,
 *   total: total failures across all buckets,
 *   unit: time unit ('minute', 'hour', 'day', etc.),
 *   graphData: [{x: timestamp, y: count, error: msg, ...}]
 * }
 */
function prepareFailedData(graphDataFail) {
    const dates = getMinMaxDateFromObject(graphDataFail);
    const unit = getDateIntervalForChart(new Date(dates.minDate), new Date(dates.maxDate));
    const useHourlyKey = (['minute', 'hour'].includes(unit));

//    console.log('useHourlyKey', useHourlyKey);
//    console.log('useHourlyKey', useHourlyKey);
    let tmpData = [];
    let result = {max: 0, total: 0, unit: unit, graphData: []};
    // Group failures by time interval
    for (let i = 0; i < graphDataFail.length; i++) {
        const key = useHourlyKey ?
            roundMinutes(graphDataFail[i].performed) :
            getLocalDatePartFromString(graphDataFail[i].performed);

        if (!tmpData[key]) {
//            mostFrequentlyErrors[key] = [];
//            mostFrequentlyErrors[key][errKey] = graphDataFail[i].seq;
            tmpData[key] = {
                count: graphDataFail[i].seq,
                from: graphDataFail[i].performed,
                until: graphDataFail[i].until,
                error: graphDataFail[i].error,
                errSeq: graphDataFail[i].seq,
            };
        } else {
//            mostFrequentlyErrors[key][errKey] = !mostFrequentlyErrors[key][errKey] ? graphDataFail[i].seq :
//                mostFrequentlyErrors[key][errKey] + graphDataFail[i].seq;
            tmpData[key].count += graphDataFail[i].seq;
            tmpData[key].until = graphDataFail[i].until;
            if (graphDataFail[i].seq > tmpData[key].errSeq) {
                tmpData[key].error = graphDataFail[i].error;
                tmpData[key].errSeq = graphDataFail[i].seq;
            }
        }
    }

    const myKeys = Object.keys(tmpData).sort();
//    console.log('myKeys', myKeys);
//    console.log('tmpData', tmpData);

    for (let i in myKeys) {
        const key = myKeys[i];
        const performed = useHourlyKey ? getCuttedDateString(key) : setMidday(key);
        result.graphData.push({
            x: performed,
            y: tmpData[key].count,
            from: tmpData[key].from,
            until: tmpData[key].until,
            error: tmpData[key].error,
            errSeq: tmpData[key].errSeq,

        });
        result.max = Math.max(result.max, tmpData[key].count);
        result.total += tmpData[key].count;
    }

    return result;
}
