/* /lib/js/ui-helpers.js */
/*!
 * UI Helper & Utility Functions for Website Checkups Plugin v1.2.0
 * https://website-checkups.com
 * Extracted from admin.js and charts.js during refactoring
 * Released under the GPLv2 license
 */

/**
 * Mapping of continent IDs to continent names
 * @constant {Object<number, string>}
 */
const continentMapping = {
    1: 'Asia',
    2: 'Europe',
    3: 'Africa',
    4: 'Oceania',
    5: 'South America',
    6: 'North America',
    7: 'Antarctica'
};

/**
 * Display order for continents in dropdown lists
 * @constant {number[]}
 */
const continentOrder = [2, 6, 1, 5, 3, 4, 7];

// ===================================
// DROPDOWN & LIST GENERATION
// ===================================

/**
 * Generates HTML options for a dropdown list
 * @param {Array} dropdown_data - Array of data objects to populate the dropdown
 * @param {string|null} default_txt - Text for the default/placeholder option
 * @param {string} data_attr - Attribute name to extract text from
 * @param {*} [select_id=null] - ID of the option to mark as selected
 * @param {boolean} [short_data=true] - Whether to sort data alphabetically
 * @returns {string} HTML string of option elements
 */
function generate_dropdown_list(dropdown_data, default_txt, data_attr, select_id = null, short_data = true) {
    const options = [];

    if (default_txt !== null) {
        options.push('<option value="">' + default_txt + '</option>');
    }

    const tmp_dropdown_data = short_data ? short_data_alphabetically(dropdown_data, data_attr) : dropdown_data;
    tmp_dropdown_data.forEach(item => {
        const value = item?.id ?? item?.value;
        const text = item?.attributes && item?.attributes[data_attr] ?
            (item.attributes[data_attr]?.text ?? item.attributes[data_attr]) : item?.text;

        const selectedAttribute = (value === select_id) ? ' selected' : '';
        options.push('<option value="' + value + '" ' + selectedAttribute + '>' + text + '</option>');
    });

    return options.join('');
}

/**
 * Generates HTML options for period selection dropdown
 * @param {Array} data - Array of period objects with value, text, and price
 * @param {*} select_id - ID of the option to mark as selected
 * @returns {string} HTML string of option elements
 */
function get_period_list(data, select_id) {
    const options = [];

    data.forEach(item => {
        const selectedAttribute = (item.value === select_id) ? ' selected' : '';
        options.push('<option value="' + item.value + '" ' + selectedAttribute + '>' + item.text +
            (item.price > 0 ? ', €' + format_num(item.price, 0) : '') + '</option>');
    });

    return options.join('');
}

/**
 * Formats subscription text with check frequency information
 * @param {string} name - Subscription name
 * @param {number} max - Maximum number of checks
 * @param {number} frequency - Check frequency in minutes
 * @returns {string} Formatted subscription description
 */
function get_subscription_text(name, max, frequency) {
    const plural = max > 1 ? 's' : '';
    const period = frequency >= 1440 ? 1440 : frequency === 1 ? 1 : 60;
    const period_str = period === 1440 ? 'daily' : period === 1 ? 'every minute' : 'hourly';
    const times = max * period / frequency;

    return `${name}, up to ${times} check${plural} ${period_str}`;
}

/**
 * Generates grouped dropdown list for subscription products (free and paid)
 * @param {Array} dropdown_data - Array of subscription product objects
 * @param {number|null} [selected_id=null] - ID of the subscription to mark as selected
 * @returns {string} HTML string with optgroup elements for free and paid subscriptions
 */
function generate_subscription_lst(dropdown_data, selected_id = null) {
    const options_free = [];
    const options_paid = [];

    dropdown_data.forEach(sub_product => {
        const attr = sub_product.attributes;
        const text = get_subscription_text(attr.name, parseInt(attr.max), parseInt(attr.frequency));
        const selected = (parseInt(sub_product.id) === selected_id) ? ' selected' : '';
        const item_option = `<option value="${sub_product.id}"${selected}>${text}</option>`;

        if (attr.name === 'Free') {
            options_free.push(item_option);
        } else {
            options_paid.push(item_option);
        }
    });

    return '<optgroup label="Free">' + options_free.join('') + '</optgroup>' +
        '<optgroup label="Paid subscriptions">' + options_paid.join('') + '</optgroup>';
}

/**
 * Generates dropdown options for subscription variants
 * @param {Array} dropdown_data - Array of subscription data
 * @param {number} [abo_id=1] - Subscription ID to get alternatives for
 * @param {number|null} [selected_id=null] - Alternative ID to mark as selected
 * @returns {string} HTML string of option elements
 */
function generate_subscription_options(dropdown_data, abo_id = 1, selected_id = null) {
    const options = [];
    const dData = dropdown_data.find(item => item.id == abo_id).attributes.checkAlternatives;

    dData.forEach(variant => {
        const optionText = variant.text;
        const selected = variant.max == selected_id ? ' selected' : '';
        options.push(`<option value="${variant.max}"${selected}>${optionText}</option>`);
    });

    return options.join('');
}

/**
* Generates grouped dropdown list with custom option text formatting
* @param {Array} dropdownData - Array of data objects
* @param {string[]} [optionTextKeys=[]] - Array of attribute keys to use for option text
* @param {boolean} [show_bracket=false] - Whether to wrap last value in brackets
* @param {Array} [selectedIds=[]] - Array of IDs to mark as selected
* @returns {string} HTML string with optgroup elements grouped by notification type
*/
function generate_grouped_dropdown_list(dropdownData, optionTextKeys = [], show_bracket = false, selectedIds = []) {
    const optionGroups = {};

    dropdownData.forEach(data => {
        const attributes = data?.attributes;
        const notificationType = attributes?.notificationType;

        let optionTextParts = optionTextKeys.map(key => attributes[key] || '').filter(part => part);

        if (show_bracket && optionTextParts.length > 1) {
            const lastValue = optionTextParts.pop(); // Extract last value
            optionTextParts.push(' (' + lastValue + ')');
        }

        // Add verified status symbol ONLY if verifiedAt attribute exists (notifications only!)
        let optionText = optionTextParts.join(' ');

        // Check if this is notification data (has verifiedAt attribute)
        if ('verifiedAt' in attributes) {
            const isVerified = !!attributes?.verifiedAt;
            const statusSymbol = isVerified ? ' ✓' : ' ✗';
            optionText += statusSymbol;
        }
        
        const selected = selectedIds.includes(data.id) ? ' selected' : '';

        // Add data-verified attribute only for notifications
        let dataAttrs = '';
        if ('verifiedAt' in attributes || 'notVerified' in attributes) {
            const isVerified = !!attributes?.verifiedAt;
            dataAttrs = ' data-verified="' + isVerified + '"';
        }

        const itemOption = '<option value="' + data.id + '"' + dataAttrs + selected + '>' + optionText + '</option>';
        
        if (!optionGroups[notificationType]) {
            optionGroups[notificationType] = [];
        }
        optionGroups[notificationType].push(itemOption);
    });

    const result = [];
    for (const type in optionGroups) {
        result.push('<optgroup label="' + type.charAt(0).toUpperCase() + type.slice(1) + '">');
        result.push(optionGroups[type].join(''));
        result.push('</optgroup>');
    }

    return result.join('');
}

// ===========
// FORMATTING
// ===========
/**
* Formats time value into human-readable string
* @param {number} value - Time value in minutes
* @returns {string} Formatted time string (e.g., "2 hours", "3 days")
*/
function format_time(value) {
    if (value < 60) {
        return value + ' minute' + (value > 1 ? 's' : '');
    } else if (value < 1440) {
        const hours = Math.floor(value / 60);
        return hours + ' hour' + (hours > 1 ? 's' : '');
    } else if (value < 10080) {
        const days = Math.floor(value / 1440);
        return days + ' day' + (days > 1 ? 's' : '');
    } else if (value < 43200) {
        const weeks = Math.floor(value / 10080);
        return weeks + ' week' + (weeks > 1 ? 's' : '');
    } else {
        const months = Math.floor(value / 43200);
        return months + ' month' + (months > 1 ? 's' : '');
    }
}

/**
 * Formats number with locale-specific formatting
 * @param {number} number - Number to format
 * @param {number} [fractionDigits=3] - Number of decimal places (0 for integers)
 * @returns {string} Formatted number string
 */
function format_num(number, fractionDigits= 3) {
    const options = fractionDigits === 0 ? {} : { minimumFractionDigits: fractionDigits };
    return new Intl.NumberFormat(navigator.language, options).format(
        number,
    );
}

/**
 * Formats date string into localized format
 * @param {string} dateString - ISO date string
 * @param {boolean} [withYear=true] - Whether to include year in output
 * @returns {string|null} Formatted date string or null if invalid
 */
function format_date(dateString, withYear=true) {
    if (!dateString) {
        return;
    }
    const dateTime = new Date(dateString);

    if (isNaN(dateTime.getTime())) {
        console.error('Error:', dateString);
        return null;
    }

    const day = String(dateTime.getDate()).padStart(2, '0');
    const monthShort = dateTime.toLocaleString(localLanguage ?? 'default', {month: 'short'});
    const year = withYear ? ' ' + dateTime.getFullYear() : '';

    const hours = String(dateTime.getHours()).padStart(2, '0');
    const minutes = String(dateTime.getMinutes()).padStart(2, '0');

    const hasTime = dateString.trim().length > 10;

    if (hasTime) {
        return `${monthShort} ${day}${year} at ${hours}:${minutes}`;
    } else {
        return `${monthShort} ${day}${year}`;
    }
}

/**
 * Formats downtime duration into compact human-readable string
 * @param {number} minutes - Downtime in minutes
 * @returns {string} Formatted downtime string (e.g., "2h 30min", "3d 5h")
 */
function format_downtime(minutes) {
    if (!minutes || minutes === 0) return '';

    if (minutes < 60) {
        return minutes + ' min';
    } else if (minutes < 1440) {
        const hours = Math.floor(minutes / 60);
        const remainingMinutes = minutes % 60;
        return remainingMinutes > 0 ? `${hours}h ${remainingMinutes}min` : `${hours}h`;
    } else {
        const days = Math.floor(minutes / 1440);
        const remainingHours = Math.floor((minutes % 1440) / 60);
        return remainingHours > 0 ? `${days}d ${remainingHours}h` : `${days}d`;
    }
}

// ===================================
// SUBSCRIPTION & PRICE HELPERS
// ===================================

/**
 * Gets subscription attributes by ID
 * @param {Object} product_data - Product data object containing subscription data
 * @param {number|string} search_id - Subscription ID to search for
 * @returns {Object|undefined} Subscription attributes or undefined if not found
 */
function get_subscription_attr(product_data, search_id) {
    return product_data.data.find(item => item.id == search_id)?.attributes;
}

/**
 * Generates and displays price description in subscription info element
 * @param {number} price - Subscription price
 * @param {string} period - Billing period (e.g., "1y", "1m")
 * @param {string} inv_period - Invoice period description
 * @returns {void}
 */
function get_price_desc(price, period, inv_period)
{
    const element_subscription_info = getElById('element_subscription_info');
    let res = '';
    const monthes = period == '1y' ? 12 : parseInt(period.substring(0, 1));
    const cost_year = price / monthes * 12;
    const cost_month = Math.round(price / monthes);

    if (price <= 0) {
        res = 'This subscription is and remains permanently <strong>free</strong> for you.';
    } else {
        res = 'This subscription will cost <strong>' + format_num(price, 0) + ' EUR </strong>*. ';
        if (monthes < 12) {
            res += 'It is <strong>' + format_num(cost_year, 0) + ' EUR</strong> * annually';
            if (monthes > 1) {
                res += ' or around <strong>' + format_num(cost_month, 0) + ' EUR</strong> * per month.'
            } else {
                res += '.';
            }
        } else if (monthes == 12) {
            res += 'It is around <strong>' + format_num(cost_month, 0) + ' EUR</strong> * per month.';
        }
        res += ' Billing takes place ' + inv_period.toLowerCase() + '.<br/>* Plus value added tax.';
    }
    element_subscription_info.innerHTML = res;
}

/**
 * Updates subscription information display
 * @param {Array} product_list - List of available subscription products
 * @param {Object} current_item - Currently selected subscription item
 * @param {string} [period='1y'] - Selected billing period
 * @returns {void}
 */
function subscription_info(product_list, current_item, period='1y') {
    const requests_info = getElById('availableRequestTypes') ?? getElById('availableRequestTypesReg');
    const selected_product = product_list?.data.find(item => parseInt(item.id) === parseInt(current_item));
    const period_data = selected_product?.attributes?.priceOption.find(e => e.value == period);

    if (requests_info) {
        const req_types = requests_info.innerHTML = selected_product.attributes.checkRequests.map(e => e.name);
        let req_string = '<ul style="list-style: disc">\n';
        req_types.forEach(name => {
            req_string += '<li>' + name + '</li>\n';
        });
        req_string += '</ul>';
        requests_info.innerHTML = req_string;
    }
    get_price_desc(period_data.price, period, period_data.text);
}

/**
 * Generates and populates filter dropdown with items
 * @param {Array} dropdown_items - Array of items to add as options
 * @param {HTMLSelectElement} dropdown_element - Dropdown element to populate
 * @param {Object|null} [data_info=null] - Optional object mapping item keys to tooltip text
 * @returns {void}
 */
function generate_filter_dropdown(dropdown_items, dropdown_element, data_info = null) {
    dropdown_items.forEach(item => {
        const option = document.createElement('option');
        option.value = make_lowercase_txt(item);
        option.textContent = item;
        if (data_info && data_info[make_lowercase_txt(item)]) {
            option.setAttribute('data-tooltip', data_info[make_lowercase_txt(item)]);
        }
        dropdown_element.appendChild(option);
    });
}

// ===================================
// STATUS & PERFORMANCE (from admin.js)
// ===================================

/**
 * Converts status info object to human-readable text
 * @param {Object} statusInfo - Status information object with 'short' property
 * @returns {string} Status text: 'Online', 'Warning', 'Offline', or 'Unknown'
 */
function get_status_text(statusInfo) {
    const status = statusInfo?.short?.toLowerCase();
    if (status === 'ok') return 'Online';
    if (status === 'warning') return 'Warning';
    if (status === 'failure') return 'Offline';
    return 'Unknown';
}

/**
 * Converts status info object to CSS class
 * @param {Object} statusInfo - Status information object with 'short' property
 * @returns {string} CSS class: 'text-success', 'text-warning', or 'text-danger'
 */
function get_status_class(statusInfo) {
    const status = statusInfo?.short?.toLowerCase();
    if (status === 'ok') return 'text-success';
    if (status === 'warning') return 'text-warning';
    return 'text-danger';
}

/**
 * Calculates performance indicator for comparison display
 * @param {number} lastTime - Most recent performance time
 * @param {number} avgTime - Average performance time
 * @returns {Object} Performance indicator object with show, class, icon, percent, tooltipText properties
 */
function calculate_performance_indicator(lastTime, avgTime) {
    if (!lastTime || !avgTime || lastTime === 0 || avgTime === 0) {
        return { show: false };
    }

    // Check absolute difference threshold (0.005 seconds)
    const absoluteDifference = Math.abs(avgTime - lastTime);
    if (absoluteDifference < 0.005) {
        return { show: false };
    }

    const performanceChange = (avgTime - lastTime) / avgTime;
    const percentChange = Math.abs(performanceChange * 100);
    const isBetter = performanceChange >= 0.3;
    const showIndicator = Math.abs(performanceChange) >= 0.3; // 30% threshold

    if (!showIndicator) {
        return { show: false };
    }

    return {
        show: true,
        class: isBetter ? 'performance-better' : 'performance-worse',
        icon: isBetter ? '↑' : '↓',
        percent: Math.round(percentChange),
        tooltipText: `${Math.round(percentChange)}% ${isBetter ? 'better' : 'worse'} than average`
    };
}

/**
 * Finds the checkup with most recent time (fastest or slowest)
 * @param {Array} [checkups=[]] - Array of checkup objects
 * @param {string} mode - Mode: 'fast' for fastest, 'slow' for slowest
 * @returns {Object|null} Checkup object with most recent time or null
 */
function get_most_recent_time(checkups = [], mode) {
    let mostRecentTime = null;

    checkups.forEach(checkup => {
        if (mode === 'fast') {
            if (mostRecentTime === null ||
                checkup.time < mostRecentTime.time ||
                (checkup.time === mostRecentTime.time && new Date(checkup.performed) > new Date(mostRecentTime.performed))) {
                mostRecentTime = checkup;
            }
        } else if (mode === 'slow') {
            if (mostRecentTime === null ||
                checkup.time > mostRecentTime.time ||
                (checkup.time === mostRecentTime.time && new Date(checkup.performed) > new Date(mostRecentTime.performed))) {
                mostRecentTime = checkup;
            }
        }
    });

    return mostRecentTime;
}

// ===================================
// STRING & ARRAY UTILITIES
// ===================================

/**
 * Converts HTML option string to array of values
 * @param {string} htmlString - HTML string containing option elements
 * @returns {string[]} Array of option values
 */
function html_string_to_array(htmlString) {
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = htmlString;
    return Array.from(tempDiv.querySelectorAll('option')).map(option => option.value);
}

/**
 * Sorts data array alphabetically by specified attribute
 * @param {Array} data - Array of objects to sort
 * @param {string} sortType - Attribute name to sort by
 * @returns {Array} Sorted array
 */
function short_data_alphabetically(data, sortType) {
    return data.sort((a, b) => {
        if (a.attributes[sortType] < b.attributes[sortType]) {
            return -1;
        }
        if (a.attributes[sortType] > b.attributes[sortType]) {
            return 1;
        }
        return 0;
    });
}

/**
 * Converts string to lowercase with first word capitalized
 * @param {string} searchString - String to convert
 * @returns {string} Converted string in camelCase style
 */
function make_lowercase_txt(searchString) {
    return searchString
        .split(' ')
        .map((word, index) =>
            index === 0
                ? word.toLowerCase()
                : word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
        )
        .join('');
}

/**
 * Adds tooltip text to input element
 * @param {HTMLElement} input_element - Input element to add tooltip to
 * @param {string} tooltip_data - Tooltip text content
 * @returns {void}
 */
function add_tooltip_txt(input_element, tooltip_data) {
    const parentElement = getParent(input_element, 2);
    const nested_tooltip = parentElement.querySelector('.info_tooltip');
    nested_tooltip.setAttribute('data-info', tooltip_data);
}

/**
 * Filters dropdown items based on data attributes
 * @param {Array} data - Array of data items
 * @param {HTMLSelectElement} dropdown_element - Dropdown element to filter
 * @param {string} filter_key - Attribute key to filter by
 * @returns {void}
 */
function filter_dropdown_items(data, dropdown_element, filter_key) {
    const notificationTypes = new Set(
        data.map(item => item.attributes[filter_key])
    );

    Array.from(dropdown_element.options).forEach(option => {
        if (option.value && !notificationTypes.has(option.value)) {
            option.style.display = 'none';
        } else {
            option.style.display = '';
        }
    });
}

/**
 * Checks if value is contained in list
 * @param {*} value - Value to search for
 * @param {Array} list - Array of objects with id property
 * @returns {boolean} True if value exists in list
 */
function is_contained(value, list) {
    return list.filter(t => t.id == value).length > 0;
}

// ==============
// DOM UTILITIES
// ==============

/**
 * Gets element by ID
 * @param {string} id - Element ID
 * @returns {HTMLElement|null} Element or null if not found
 */
function getElById(id) {
    return document.getElementById(id);
}

/**
 * Gets parent element at specified level
 * @param {HTMLElement} el - Starting element
 * @param {number} [level=1] - Number of parent levels to traverse
 * @returns {HTMLElement} Parent element
 */
function getParent(el, level=1) {
    for (i=level; i>0; i--) {
        el = el.parentElement;
    }
    return el;
}

/**
 * Gets all elements matching selector
 * @param {string} sel - CSS selector
 * @returns {NodeList} List of matching elements
 */
function getAllElements(sel) {
    return document.querySelectorAll(sel);
}

/**
 * Sets CSS classes on element
 * @param {HTMLElement} element - Element to modify
 * @param {string} toRemove - Class to remove
 * @param {string} [toAdd=''] - Class to add
 * @returns {void}
 */
function setClass(element, toRemove, toAdd='') {
    if (!element) {
        return;
    }
    if (toRemove !== '') {
        element.classList.remove(toRemove);
    }
    if (toAdd !== '') {
        element.classList.add(toAdd);
    }
}

/**
 * Toggles 'd-none' class based on active state
 * @param {HTMLElement} el - Element to toggle
 * @param {boolean} active - Whether element should be visible
 * @returns {void}
 */
function toggleNoneClass(el, active) {
    setClass(el, active ? 'd-none' : '', !active ? 'd-none' : '');
}

// ===================================
// TIMESPAN HELPER
// ===================================

/**
 * Formats date range or single date as timespan string
 * @param {string} from - Start date string
 * @param {string} until - End date string
 * @param {boolean} [withYear=true] - Whether to include year in output
 * @returns {string} Formatted timespan string
 */
function getTimespan(from, until, withYear=true) {
    if (from === until) {
        return format_date(from, withYear);
    } else {
        return format_date(from, withYear) + ' - ' + format_date(until, withYear);
    }
}

// ===================================
// DATE UTILITIES
// ===================================

/**
 * Sets time to midday (12:00) for given date
 * @param {string|Date} date - Date to set to midday
 * @returns {string} ISO date string with time set to 12:00:00Z
 */
function setMidday(date) {
    const d = date.toString().split('-');
    const tmpDate = new Date(d[0], d[1] - 1, d[2], 12, 0, 0);
    return getCuttedDateString(tmpDate);
}

/**
 * Converts date to ISO string and cuts to 16 characters with Z suffix
 * @param {string|Date} date - Date to convert
 * @returns {string} Cut ISO date string (YYYY-MM-DDTHH:MMZ)
 */
function getCuttedDateString(date) {
    if (typeof date == 'string') {
        date = new Date(date);
    }
    return date.toISOString().substring(0, 16) + 'Z';
}

/**
 * Gets local date part from date string based on timezone
 * @param {string} dateString - ISO date string
 * @returns {string} Date string in YYYY-MM-DD format
 */
function getLocalDatePartFromString(dateString) {
    const res = new Date(dateString).toLocaleDateString(
        'it-IT', {timeZone: localTimeZone}
    ).substring(0, 10).split('/');

    return res[2] + '-' + res[1] + '-' + res[0];
}

/**
 * Rounds date to nearest hour
 * @param {string} dateString - ISO date string
 * @returns {string} ISO date string rounded to nearest hour
 */
function roundMinutes(dateString)
{
    date = new Date(dateString);
    date.setHours(date.getHours() + Math.round(date.getMinutes()/60));
    date.setMinutes(0, 0, 0); // Resets also seconds and milliseconds

    return getCuttedDateString(date);
}

// ===================================
// STRING & FORMATTING (from charts.js)
// ===================================

/**
 * Converts request type to service type description
 * @param {string} requestType - Request type code (e.g., 'FTP', 'TCP', 'DNS')
 * @returns {string} Service type description (e.g., 'FTP server', 'port', 'server', 'website')
 */
function getServiceType(requestType) {
    let res = 'website';
    switch (requestType) {
        case 'FTP':
            res = 'FTP server';
            break;
        case 'TCP':
        case 'UDP':
            res = 'port'
            break;
        case 'MAIL':
        case 'DNS':
        case 'PING':
            res = 'server';
            break;
    }
    return res;
}

/**
 * Capitalizes first character of string
 * @param {*} val - Value to capitalize
 * @returns {string} String with first character uppercased
 */
function upperFirst(val) {
    return String(val).charAt(0).toUpperCase() + String(val).slice(1);
}

/**
 * Splits string into chunks of max 70 characters at word boundaries
 * @param {string} str - String to split
 * @returns {string[]|null} Array of string chunks or null
 */
function splitStringInChunks(str) {
    return str.match(/.{1,70}(?:\b|$)/g);
}

/**
 * Formats datetime as hours:minutes with optional date
 * @param {string|Date} datetime - DateTime to format
 * @param {boolean} [withDate=false] - Whether to include date in output
 * @returns {string} Formatted time string (HH:MM or DD.MM.YYYY HH:MM)
 */
function printHoursMinutes(datetime, withDate=false) {
    if (typeof datetime == 'string') {
        datetime = new Date(datetime);
    }
    return (withDate ? datetime.toLocaleDateString(localLang) + ' ' : '') +
        ('0' + datetime.getHours()).slice(-2) + ':' + ('0' + datetime.getMinutes()).slice(-2);
}

/**
 * Formats number with locale-specific formatting (duplicate of format_num)
 * @param {number} number - Number to format
 * @param {number} [fractionDigits=3] - Number of decimal places
 * @returns {string} Formatted number string
 */
function formatNumbers(number, fractionDigits= 3) {
    const options = fractionDigits == 0 ? {} : { minimumFractionDigits: fractionDigits };
    return new Intl.NumberFormat(localLang, options).format(
        number,
    );
}

// ====================
// TOAST NOTIFICATIONS
// ====================

/**
 * Handles API response and shows appropriate toast notification
 * @param {Object} obj - Response object from API
 * @param {string} [success_text='Your data has been processed successfully'] - Success message text
 * @returns {void}
 */
function handle_toast(obj, success_text='Your data has been processed successfully') {
    if (obj?.errors) {
        generate_multiple_toast(obj?.errors);
    } else if (!obj?.meta) {
        show_toast(true, success_text, 1);
    } else {
        show_toast(true, obj.meta?.notice, 1);
    }
    close_loader_safe();
}

/**
 * Shows toast notification
 * @param {boolean} result - Success state (true for success, false for error)
 * @param {string} msg - Message to display
 * @param {number} [toast_delay=3] - Display duration in seconds
 * @returns {void}
 */
function show_toast(toast_status, toast_msg, toast_delay = 1) {
    clear_all_toast();
    if (toast_msg !== undefined) {
        fetch_template('template_toast.html')
            .then(template_data => {
                const timestamp = Date.now().toString(36);
                const randomPart = Math.random().toString(36).substring(2, 8);
                const unique_id = timestamp + '-' + randomPart;

                let toast_color = 'danger';

                if (toast_status) {
                    toast_color = 'success';
                }

                if (toast_status === 'light') {
                    toast_color = 'Light';
                }

                template_data = template_data.replace('{{toast_id}}', unique_id);
                template_data = template_data.replace('{{toast_color}}', toast_color);
                template_data = template_data.replace('{{toast_msg}}', toast_msg);

                const existing_toasts = getAllElements('.toast .toast-body');
                for (let i = 0; i < existing_toasts.length; i++) {
                    if (existing_toasts[i].textContent.trim() === toast_msg.trim()) {
                        return;
                    }
                }
                getElById('toast_holder').innerHTML += template_data;

                const toastElList = [].slice.call(getAllElements('.toast'));
                const toastList = toastElList.map(function (toastEl) {
                    const toast = new bootstrap.Toast(toastEl);
                    toastEl.addEventListener('hidden.bs.toast', () => {
                        toastEl.remove();
                    });

                    return toast;
                });
                toastList.forEach(toast => toast.show({delay: toast_delay * 1000}));

            }).catch(error => {
            console.error('Toast template fetch error:', error);
        });
    }
}

/**
* Clears all toast notifications
* @returns {void}
*/
function clear_all_toast() {
    getElById('toast_holder').innerHTML = '';
}

/**
 * Generates multiple toast notifications from error items
 * @param {Array} items - Array of error items with detail and source properties
 * @returns {void}
 */
function generate_multiple_toast(items) {
    if (items?.length > 0) {
        const ol = document.createElement('ol');
        ol.style.margin = '0px';

        items.forEach(item => {
            if (item?.detail) {
                const li = document.createElement('li');
                li.textContent = item?.detail;
                ol.appendChild(li);

                if (item?.source) {
                    const tmp_item_name = item.source?.pointer.replace(replace_string, '');
                    generate_field_error(Object.values(wc_form_data).flat().find(item => item?.field_name === tmp_item_name).field_item);
                }
            }
        });
        if (ol.childElementCount > 0) {
            show_toast(false, ol.outerHTML, 1);
        }
    }
}

// =================
// LOADER FUNCTIONS
// =================

/**
 * Shows animated loading indicator
 * @param {boolean} [transparent_bg=true] - Whether to use transparent background
 * @param {string} [loader_id='page_loader'] - ID for loader element
 * @param {string} [loader_location='.loader_block'] - CSS selector for loader container
 * @returns {void}
 */
function show_loader(transparent_bg = true, loader_id = 'page_loader', loader_location = '.loader_block') {
    if (getElById(loader_id)) {
        return;
    }
    const desiredHeight = 100;
    const originalWidth = 200;
    const originalHeight = 200;
    const scaleFactor = desiredHeight / originalHeight;
    const desiredWidth = originalWidth * scaleFactor;

    const loaderDiv = document.createElement('div');
    loaderDiv.id = loader_id;
    loaderDiv.className = 'loader';
    if (transparent_bg) {
        setClass(loaderDiv, '', 'transparent_bg');
    }

    const svgLoader = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svgLoader.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
    svgLoader.setAttribute('viewBox', '0 0 200 200');
    svgLoader.setAttribute('width', desiredWidth);
    svgLoader.setAttribute('height', desiredHeight);

    const rect1 = createAnimatedRect(25 * scaleFactor, 85 * scaleFactor, scaleFactor);
    const rect2 = createAnimatedRect(85 * scaleFactor, 85 * scaleFactor, scaleFactor);
    const rect3 = createAnimatedRect(145 * scaleFactor, 85 * scaleFactor, scaleFactor);

    svgLoader.appendChild(rect1);
    svgLoader.appendChild(rect2);
    svgLoader.appendChild(rect3);

    loaderDiv.appendChild(svgLoader);

    const wrapElement = document.querySelector(loader_location);
    if (wrapElement !== null) {
        wrapElement.insertBefore(loaderDiv, wrapElement.firstChild);
        document.querySelector('body').classList.add('no_scroll');
    }
}

/**
 * Removes loading indicator
 * @param {string} [loader_id='page_loader'] - ID of loader element to remove
 * @returns {void}
 */
function close_loader(loader_id = 'page_loader') {
    const loader = getElById(loader_id);
    if (loader) {
        loader.remove();
        document.querySelector('body').classList.remove('no_scroll');
    }
}

// ============================
// FORM/FIELD HELPER FUNCTIONS
// ============================

/**
 * Clears all fields in form
 * @param {Array} form_field - Array of field objects
 * @returns {boolean} False if form_field is empty
 */
function clear_all_field(form_field) {
    if (!form_field || Object.keys(form_field).length === 0) {
        return false;
    }

    form_field.forEach(item => {
        const field = item.field_item;
        if (field?.value) {
            field.value = '';
        }
        if (field?.checked) {
            field.checked = false;
        }
    });
}

/**
 * Sets up error clearing listeners on fields
 * @param {Array} fields - Array of field objects
 * @returns {void}
 */
function field_error_clear(fields) {
     // Guard: Return early if fields is undefined or not an array
     if (!fields || !Array.isArray(fields)) {
         console.warn('field_error_clear: fields parameter is not an array');
         return;
     }

    fields.forEach(item => {
         // Skip if field_item is null/undefined (field doesn't exist in DOM)
         if (!item || !item.field_item) {
             return;
         }

        const field = item.field_item;
        const removeErrorClass = () => {
            setClass(field, 'is-invalid');
        };

        field.addEventListener('focus', removeErrorClass);
        field.addEventListener('change', removeErrorClass);
    });
}

/**
 * Removes error class from all invalid inputs
 * @returns {void}
 */
function remove_all_error_class() {
    const error_input = getAllElements('.is-invalid');
    error_input.forEach(input => {
        setClass(input, 'is-invalid');
    });
}

/**
 * Validates required fields and displays errors
 * @param {Array} fields - Array of field objects to validate
 * @returns {boolean} True if validation errors found
 */
function field_error_check(fields) {
    let has_error = false;
    field_error_clear(fields);
    let tmp_required_field = [];

    fields.forEach(field => {
        const isFieldRequired = (field?.field_required !== false);
        if ((field?.field_item.getAttribute('type') !== 'hidden' && isFieldRequired)) {
            if (field?.field_item?.value.trim() === '' || (field?.field_item?.checked === false && field?.field_item?.getAttribute('type') === 'checkbox')) {
                generate_field_error(field?.field_item);
                has_error = true;
                tmp_required_field.push(field);
            }
        }
    });
    if (has_error) {
        generate_multiple_toast(tmp_required_field);
    }
    return has_error;
}

/**
 * Adds error class to field
 * @param {HTMLElement} field - Field element to mark as invalid
 * @returns {void}
 */
function generate_field_error(field) {
    setClass(field, '', 'is-invalid');
}

/**
 * Enables all fieldset elements on page
 * @returns {void}
 */
function enable_all_fieldset() {
    const all_fieldset = getAllElements('body fieldset');
    all_fieldset.forEach(single_fieldset => {
        single_fieldset.disabled = false;
        if (single_fieldset.querySelector('.spinner-border')?.classList) {
            single_fieldset.querySelector('.spinner-border').classList.add('d-none');
        }
    });
}

/**
 * Toggles disabled state of fieldset containing button
 * @param {HTMLElement} btn_element - Button element within fieldset
 * @returns {void}
 */
function toggle_disable_frm(btn_element) {
    const parent_fieldset = btn_element.closest('fieldset');
    const spinner = btn_element.querySelector('.spinner-border');

    if (!parent_fieldset.disabled) {
        parent_fieldset.disabled = true;
        setClass(spinner, 'd-none');
    } else {
        parent_fieldset.disabled = false;
        setClass(spinner, '', 'd-none');
    }
}

/**
 * Clears all form fieldsets by enabling them and hiding spinners
 * @returns {void}
 */
function clear_all_form() {
    const field_sets = getAllElements('fieldset');

    field_sets.forEach(fieldset => {
        const spinners = getAllElements('.spinner-border');
        fieldset.disabled = false;
        spinners.forEach(spinner => {
            setClass(spinner, '', 'd-none');
        })
    });
}

/**
 * Toggles extended/advanced fields based on request type
 * @param {string} req_type - Request type to filter fields
 * @returns {void}
 */
function toggle_extended(req_type) {
    const btn = getElById('sitecheckups_block_advanced_btn');
    const active = btn && btn.getAttribute('aria-pressed') == 'true';

    wc_form_data['sitecheckups'].forEach(form_data => {
        const el = getParent(form_data.field_item, form_data.field_type == 'checkbox' ? 2 : 3);
        if (form_data.disable_for) {
            const disabled = form_data.disable_for.includes(req_type);
            el.classList[disabled ? 'add' : 'remove']('d-none');
        }
        if (form_data.field_advanced) {
            const state = active && (!form_data.disable_for || !form_data.disable_for.includes(req_type));
            toggleNoneClass(el, state);
        }
    });
}

/**
 * Toggles visibility of all advanced fields
 * @param {boolean} active - Whether advanced fields should be visible
 * @returns {void}
 */
function toggle_advanced_fields(active) {
    const fields_additional = getAllElements('.hidden_toggle');
    fields_additional.forEach(el => {
        toggleNoneClass(el, active);
    });

    if (active) {
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
}

/**
 * Checks if any extended fields are filled
 * @param {string} req_type - Request type to check
 * @returns {number} 1 if any extended fields are filled, 0 otherwise
 */
function filled_extended_fields(req_type) {
    const ext_fields = ['authtype', 'btoken', 'postData', 'headers', 'noChecksFrom', 'noChecksUntil', 'responseText', 'username', 'password'];
    let res = 0;
    wc_form_data['sitecheckups'].forEach(t => {
        res += ext_fields.includes(t.field_name) && t?.field_item?.value?.trim().length > 0 &&
        (!t.disable_for || !t.disable_for.includes(req_type)) ? 1 : 0;
    });

    return res > 0;
}

/**
 * Resets all form fields in modal
 * @returns {void}
 */
function reset_form_field() {
    const modal_element = getElById('modal_' + modal_name);
    const elements = modal_element.querySelectorAll('input, select, button, textarea');
    elements.forEach(element => {
        element.value = '';
    });
}

/**
 * Closes Bootstrap modal
 * @param {string} modal_id - ID of modal to close
 * @returns {void}
 */
function close_modal(modal_id) {
    const modal = getElById(modal_id);
    const activeElement = document.activeElement;
 
    // If focus is inside modal, move it outside before closing
    if (activeElement && modal.contains(activeElement)) {
        // Focus a safe element outside the modal (e.g., body)
        document.body.focus();
        // Remove focus from body immediately to prevent outline
        document.body.blur();
    }
 
    bootstrap.Modal.getInstance(modal).hide();
}

// ===================
// TABLE/UI FUNCTIONS
// ===================

/**
* Searches table rows and filters by search value
* @param {HTMLElement} tbl_id - Table element to search
* @param {string} search_value - Search query string
* @param {number[]} col_indices - Array of column indices to search in
* @returns {void}
*/
function search_tbl(tbl_id, search_value, col_indices) {
    const rows = tbl_id.getElementsByTagName('tbody')[0].getElementsByTagName('tr');

    const lowerCaseSearchValue = search_value.toLowerCase();

    for (let i = 0; i < rows.length; i++) {
        let rowContainsSearchValue = false;

        for (let j = 0; j < col_indices.length; j++) {
            const col_index = col_indices[j];
            const cell = rows[i].getElementsByTagName('td')[col_index];

            if (cell) {
                const cellText = (cell.textContent || cell.innerText).toLowerCase();

                if (cellText.indexOf(lowerCaseSearchValue) > -1) {
                    rowContainsSearchValue = true;
                    break;
                }
            }
        }

        if (rowContainsSearchValue) {
            rows[i].style.display = '';
        } else {
            rows[i].style.display = 'none';
        }
    }
}

/**
 * Links main checkbox to child checkboxes for select all functionality
 * @param {HTMLInputElement} main_check_box - Main checkbox element
 * @param {NodeList|Array} children_chk_box - Child checkbox elements
 * @returns {void}
 */
function toggle_all_check_box(main_check_box, children_chk_box) {
    main_check_box.addEventListener('change', function () {
        children_chk_box.forEach(checkbox => {
            checkbox.checked = main_check_box.checked;
        });
    });

    children_chk_box.forEach(checkbox => {
        checkbox.addEventListener('change', function () {
            if (!checkbox.checked) {
                main_check_box.checked = false;
            }
        });
    });
}

/**
 * Renders table with data using template
 * @param {string} table_id - ID of table element
 * @param {Array} data_array - Array of data items to render
 * @param {string} template_path - Path to HTML template file
 * @param {Function} mapper_fn - Function to map data to HTML (template, item, index) => html
 * @returns {Promise<void>}
 */
async function render_table(table_id, data_array, template_path, mapper_fn) {
    const table = getElById(table_id);
    const template = await fetch_template(template_path);

    table.querySelector('tbody').innerHTML = '';

    data_array.forEach((item, index) => {
        const row_html = mapper_fn(template, item, index);
        table.querySelector('tbody').innerHTML += row_html;
    });
}

/**
 * Fetches HTML template from file
 * @param {string} file_name - Template file name
 * @returns {Promise<string|boolean>} Template HTML string or false on error
 */
async function fetch_template(file_name) {
    try {
        const file_path = wc_template_loc + file_name;
        const response = await fetch(file_path);

        if (!response.ok) {
            console.error('Template fetch error for:', file_name);
        }
        return await response.text();
    } catch (error) {
        console.error('Template fetch error:', error);
        return false;
    }
}

/* replace manage_local_data for some forms */
/**
 * Prepares form fields from API field descriptions
 * @param {string} form - Form identifier
 * @returns {Promise<Array>} Array of mapped form field objects
 */
async function prepare_form_fields(form) {
    // console.log('🔍 prepare_form_fields called for:', form);
    const form_data = await manage_api_data('fielddescriptions', 'get');
    // console.log('📦 API response for fielddescriptions:', form_data);

    const form_fields = form_data.data ?
        form_data.data.filter(e => e.attributes.form === form).map(item => item.attributes) : [];

    const form_fields_mapped = form_fields ? form_fields.map(item => ({
        'field_item': null,
        'field_name': item.field,
        'field_type': item.type,
        'field_advanced': item.advanced === 1 ? 1 : null,
        'field_required': (item.required === 1),
        'values_description': item.valuesDescription,
        'disable_for': item.disableFor,
        'detail': item?.required === 1 && item?.label ? item.label + ' is required' : null
    })) : [];

    // console.log('📋 Mapped fields:', form_fields_mapped.length, 'for form:', form);
    return form_fields_mapped;
}

// ===========================
// COUNTRY/DROPDOWN FUNCTIONS
// ===========================

/**
 * Generates country dropdown with continent grouping
 * @param {Array} country_data - Array of country objects
 * @param {string|null} default_txt - Text for default option
 * @param {*} [select_id=null] - ID of country to mark as selected
 * @returns {string} HTML string with optgroup elements for continents
 */
function generate_country_dropdown_with_continents(country_data, default_txt, select_id = null) {
    const options = [];

    if (default_txt !== null) {
        options.push('<option value="">' + default_txt + '</option>');
    }

    // Gruppiere Länder nach Kontinenten
    const countriesByContinent = {};
    continentOrder.forEach(continentId => {
        countriesByContinent[continentId] = [];
    });

    // Sortiere Länder in Kontinente ein
    country_data.forEach(country => {
        const continentId = country.attributes?.continentId;
        if (continentId && countriesByContinent[continentId]) {
            countriesByContinent[continentId].push({
                id: country.id,
                name: country.attributes.name
            });
        }
    });

    // Sortiere Länder innerhalb jedes Kontinents alphabetisch
    Object.keys(countriesByContinent).forEach(continentId => {
        countriesByContinent[continentId].sort((a, b) => a.name.localeCompare(b.name));
    });

    // Generiere HTML mit optgroups
    continentOrder.forEach(continentId => {
        const countries = countriesByContinent[continentId];

        // Nur Kontinente mit Ländern anzeigen
        if (countries.length > 0) {
            const continentName = continentMapping[continentId];
            options.push('<optgroup label="' + continentName + '">');

            countries.forEach(country => {
                const selectedAttribute = (country.id === select_id) ? ' selected' : '';
                options.push('<option value="' + country.id + '"' + selectedAttribute + '>' + country.name + '</option>');
            });

            options.push('</optgroup>');
        }
    });

    return options.join('');
}

/**
 * Generates country dropdown with optional continent grouping
 * @param {Array} country_data - Array of country objects
 * @param {string|null} default_txt - Text for default option
 * @param {*} [select_id=null] - ID of country to mark as selected
 * @param {boolean} [use_continents=false] - Whether to group by continents
 * @returns {string} HTML string of option elements
 */
function generate_country_dropdown(country_data, default_txt, select_id = null, use_continents = false) {
    if (use_continents) {
        return generate_country_dropdown_with_continents(country_data, default_txt, select_id);
    }
    return generate_dropdown_list(country_data, default_txt, 'name', select_id);
}

/**
 * Sets up switch toggle between two dropdown data sets
 * @param {HTMLInputElement} switch_id - Switch element
 * @param {HTMLSelectElement} dropdown_id - Dropdown element
 * @param {string|null} [data_one=null] - HTML options for switch off state
 * @param {string|null} [data_two=null] - HTML options for switch on state
 * @returns {void}
 */
function toggle_switch(switch_id, dropdown_id, data_one = null, data_two = null) {
    switch_id.onchange = function () {
        const tmp_dropdown_value = dropdown_id.value;
        if (switch_id.checked) {
            dropdown_id.innerHTML = data_two;

            if (!html_string_to_array(data_two).includes(tmp_dropdown_value)) {
                dropdown_id.selectedIndex = 0;
            } else {
                dropdown_id.value = tmp_dropdown_value;
            }

            switch_id.value = 1;
        } else {
            dropdown_id.innerHTML = data_one;

            if (!html_string_to_array(data_one).includes(tmp_dropdown_value)) {
                dropdown_id.selectedIndex = 0;
            } else {
                dropdown_id.value = tmp_dropdown_value;
            }

            switch_id.value = 0;
        }
    };
}

/**
* Gets default country based on browser locale
* @param {Array} allList - Array of all available country objects
* @returns {string} Country code (lowercase)
*/
function getDefaultCountry(allList) {
    if (!navigator.languages || navigator.languages.length === 0) {
        return 'us'; // Fallback wenn keine Browser-Sprachen verfügbar
    }

    // PRIORITY 1: Country-Code aus Browser-Locale (z.B. de-DE -> de)
    for (let i = 0; i < navigator.languages.length; i++) {
        const locale = navigator.languages[i];

        // Extrahiere Country-Code (z.B. "de-DE" -> "de", "en-US" -> "us")
        if (locale.includes('-')) {
            const countryCode = locale.substring(3).toLowerCase();
            if (countryCode && is_contained(countryCode, allList)) {
                // console.log('🌍 Country detected from browser locale:', countryCode, '(' + locale + ')');
                return countryCode;
            }
        }
    }

    // PRIORITY 2: Direct Language-Country-Mapping (z.B. de -> de, fr -> fr, it -> it)
    for (let i = 0; i < navigator.languages.length; i++) {
        const locale = navigator.languages[i];
        const langCode = locale.substring(0, 2).toLowerCase();

        if (langCode && is_contained(langCode, allList)) {
            // console.log('🌍 Country detected from language code:', langCode);
            return langCode;
        }
    }

    // FALLBACK: US (international standard)
    // console.log('🌍 Using fallback country: us');
    return 'us';
}

// ================
// SMALL UTILITIES
// ================

/**
 * Generates unique ID with timestamp and counter
 * @returns {string} Unique ID in format "id-{timestamp}-{counter}"
 */
function generate_unique_id() {
    const timestamp = Date.now();
    uid_counter = (uid_counter + 1) % 1000;
    return `id-${timestamp}-${uid_counter.toString().padStart(3, '0')}`;
}

/**
 * Shows browser confirmation dialog
 * @param {string} confirm_msg - Message to display
 * @returns {boolean} True if user confirmed
 */
function confirm_window(confirm_msg) {
    return confirm(confirm_msg);
}

/**
 * Removes empty tooltip attributes from text
 * @param {string} text - HTML text to clean
 * @returns {string} Cleaned text
 */
function cleanEmptyToolTip(text) {
    return text.replaceAll(' class="tool_tip" data-tooltip=""', '');
}

/**
 * Creates animated SVG rectangle for loader
 * @param {number} x - X coordinate
 * @param {number} y - Y coordinate
 * @param {number} scale - Scale factor
 * @returns {SVGRectElement} Animated rectangle element
 */
function createAnimatedRect(x, y, scale) {
    const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    rect.setAttribute('fill', '#000000');
    rect.setAttribute('stroke', '#000000');
    rect.setAttribute('stroke-width', '11');
    rect.setAttribute('width', 30 * scale);
    rect.setAttribute('height', 30 * scale);
    rect.setAttribute('x', x);
    rect.setAttribute('y', y);

    const animate = document.createElementNS('http://www.w3.org/2000/svg', 'animate');
    animate.setAttribute('attributeName', 'opacity');
    animate.setAttribute('calcMode', 'spline');
    animate.setAttribute('dur', '1.7s');
    animate.setAttribute('values', '1;0;1;');
    animate.setAttribute('keySplines', '.5 0 .5 1;.5 0 .5 1');
    animate.setAttribute('repeatCount', 'indefinite');

    if (x === 25 * scale) {
        animate.setAttribute('begin', '-.4s');
    } else if (x === 85 * scale) {
        animate.setAttribute('begin', '-.2s');
    } else if (x === 145 * scale) {
        animate.setAttribute('begin', '0s');
    }

    rect.appendChild(animate);
    return rect;
}

/**
 * Counts total visible rows in a select (optgroups + options)
 * @param {HTMLSelectElement} selectElement - The select element
 * @returns {number} Total number of visible rows (headers + options)
 */
function count_select_visible_rows(selectElement) {
    if (!selectElement) return 0;

    const optgroups = selectElement.querySelectorAll('optgroup');
    const options = selectElement.querySelectorAll('option');

    // optgroup headers take up space in the select!
    return optgroups.length + options.length;
}

/**
 * Calculates adaptive size for multi-select dropdown
 * @param {number} totalVisibleRows - Total visible rows (optgroups + options)
 * @returns {number} Number of visible rows
 */
function calculate_adaptive_select_size(totalVisibleRows) {
    // Adaptive sizing without expand/collapse
    if (totalVisibleRows <= 4) return 4;
    if (totalVisibleRows <= 6) return 6;
    if (totalVisibleRows <= 8) return 8;
    if (totalVisibleRows <= 10) return 10;
    if (totalVisibleRows <= 12) return 12;

    // Maximum size for very large lists
    return 12;
}

/**
 * Adds helper text with stats for notification dropdown
 * @param {HTMLSelectElement} selectElement - The select element
 * @param {number} totalNotifications - Total number of notifications
 * @returns {void}
 */
function add_notification_helper_text(selectElement, totalNotifications) {
    // Find the correct container
    let container = selectElement.closest('.col-12, .col-6, .col-md-12, [class*="col-"]');
    if (!container) {
        console.error('❌ Could not find column container for helper text');
        return;
    }

    // Check if helper text already exists
    let existingHelper = container.querySelector('.notification-helper-text');
    if (existingHelper) {
        // console.log('ℹ️ Helper text already exists');
        return;
    }

    // Create helper text
    const helperText = document.createElement('div');
    helperText.className = 'form-text notification-helper-text mt-2';
    helperText.innerHTML = `
        <i class="bi bi-info-circle me-1"></i>
        <span>✓</span> = Verified,
        <span>✗</span> = Not verified
        <span class="text-muted ms-3">${totalNotifications} notification${totalNotifications === 1 ? '' : 's'} available</span>
    `;

    // Insert after expand button if exists, otherwise after input-group
    const expandBtn = container.querySelector('.notification-expand-btn');
    if (expandBtn) {
        expandBtn.insertAdjacentElement('afterend', helperText);
    } else {
        const inputGroup = container.querySelector('.input-group');
        if (inputGroup) {
            inputGroup.insertAdjacentElement('afterend', helperText);
        } else {
            container.appendChild(helperText);
        }
    }

    // console.log('✅ Helper text created and inserted into:', container.className);
}