<?php
/* /lib/script/manage_data.php */
if (!defined('ABSPATH')) {
    exit;
}

function webcheckups_save_data_to_db($token_data): bool
{
    $result = update_option(webcheckups_plugin_prefix . '_token', $token_data);
    if ($result) {
        return true;
    }
    return false;
}

function webcheckups_get_token_from_db()
{
    $saved_data = get_option(webcheckups_plugin_prefix . '_token');
//    echo 'TOKEN=' . $saved_data . PHP_EOL;
    if (!empty($saved_data)) {
        return $saved_data;
    }
    return false;
}

function webcheckups_sanitize_token($token) {
    // Token sollte alphanumerisch sein mit bestimmter Länge
    $token = sanitize_text_field($token);
    if (!preg_match('/^[a-zA-Z0-9\-_]{9,128}$/', $token)) {
        return false;
    }
    return $token;
}

add_action('wp_ajax_manage_admin_data', 'webcheckups_manage_admin_data');

function webcheckups_manage_admin_data(): void
{
    // Strikte Nonce-Prüfung VOR jeder Operation
    if (!isset($_POST['nonce']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), webcheckups_admin_nonce)) {
        wp_send_json_error(['msg' => 'Security check failed.']);
        wp_die();
    }

    // Capability-Check hinzufügen
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['msg' => 'Insufficient permissions.']);
        wp_die();
    }

    if (isset($_POST['post_type']) && $_POST['post_type'] == 'load_form') {
        // Felderdefinitionen aus wp_options holen
        $field_definitions = webcheckups_get_field_definitions();

        if (empty($field_definitions)) {
            wp_send_json_error('Field definitions not available');
            wp_die();
        }

        // Gruppieren nach Form
        $grouped = [];
        foreach ($field_definitions as $field) {
            $form_name = $field['attributes']['form'];
            if (!isset($grouped[$form_name])) {
                $grouped[$form_name] = ['form_name' => $form_name, 'fields' => []];
            }
            $grouped[$form_name]['fields'][] = $field['attributes'];
        }

        wp_send_json_success(array_values($grouped));
    }

    if (isset($_POST['post_type']) && $_POST['post_type'] == 'user_register') {
        $token = isset($_POST['token']) ? webcheckups_sanitize_token(sanitize_text_field(wp_unslash($_POST['token']))) : false;
        if ($token === false) {
            wp_send_json_error(['msg' => 'Invalid token format']);
            wp_die();
        }
        $notice = isset($_POST['notice']) ? sanitize_text_field(wp_unslash($_POST['notice'])) : '';

        if (webcheckups_save_data_to_db($token)) {
            wp_send_json_success([
                'msg' => [ $notice ],
            ]);
        } else {
            wp_send_json_error([ 'msg' => 'Save Token error' ]);
        }
    }
    wp_die();
}

function webcheckups_page_template($template_name, $show_modal): string
{
    $tmp_token_data = webcheckups_get_token_from_db();
//    var_dump($tmp_token_data);

    if ($template_name == 'setting') {
        if (!$tmp_token_data) {
            $form_data       = webcheckups_manage_form_data('simpleregisters');
            $form_field_data = '';
            foreach ($form_data as $single_form_item) {
                $field_attr  = $single_form_item['attributes'];
                $with_switch = false;

                if (isset($field_attr['label'])) {
                    if (strtolower($field_attr['label']) == 'language' ||
                        strtolower($field_attr['label']) == 'country') {
                        $with_switch = true;
                    }
                }

                $form_field_data .= webcheckups_generate_form_field($field_attr, $with_switch);
            }

            $template_path = file_get_contents(webcheckups_lib_dir_path . '/template/page/page_signup.html');
            $template_path = str_replace('{{form_field}}', $form_field_data, $template_path);
        } else {
            $template_path = file_get_contents(
                webcheckups_lib_dir_path . '/template/page/page_' . $template_name . '.html'
            );

            $tmp_user_form_field_data         = '';
            $tmp_registration_form_field_data = '';
            $tmp_subscription_form_field_data = '';

            $user_form_data         = webcheckups_manage_form_data('users');
            $registration_form_data = webcheckups_manage_form_data('registrations');
            $subscription_form_data = webcheckups_manage_form_data('subscriptions');

            foreach ($user_form_data as $single_user_form_item) {
                $field_attr = $single_user_form_item['attributes'];

                $tmp_user_form_field_data .= webcheckups_generate_form_field($field_attr);
            }
            $template_path = str_replace('{{form_user}}', $tmp_user_form_field_data, $template_path);

            foreach ($registration_form_data as $single_registration_form_item) {
                $field_attr = $single_registration_form_item['attributes'];

                if (isset($field_attr['label'])) {
                    if (strtolower($field_attr['label']) == 'country') {
                        $with_switch = true;
                    } else {
                        $with_switch = false;
                    }
                    $tmp_registration_form_field_data .= webcheckups_generate_form_field($field_attr, $with_switch);
                }
            }

            foreach ($subscription_form_data as $single_subscription_form_item) {
                $field_attr = $single_subscription_form_item['attributes'];

                $tmp_subscription_form_field_data .= webcheckups_generate_form_field($field_attr, false);
            }

            $template_path = str_replace('{{form_user}}', $tmp_user_form_field_data, $template_path);
            $template_path = str_replace(
                '{{form_registration}}',
                $tmp_registration_form_field_data,
                $template_path
            );
            $template_path = str_replace(
                '{{form_subscription}}',
                $tmp_subscription_form_field_data,
                $template_path
            );
        }
    } else {
        $template_path = file_get_contents(webcheckups_lib_dir_path . '/template/page/page_' .
            $template_name . '.html');
    }

    if ($show_modal) {
        $tmp_popup_form_field_data = '';
        $template_modal_path       = file_get_contents(
            webcheckups_lib_dir_path . 'template/modal/modal_' . $template_name . '.html'
        );

        $popup_form_data = webcheckups_manage_form_data($template_name);
        usort($popup_form_data, fn($a, $b) => $a['attributes']['position'] <=> $b['attributes']['position']);

        foreach ($popup_form_data as $single_popup_form_item) {
            $field_attr = $single_popup_form_item['attributes'];

            $tmp_popup_form_field_data .= webcheckups_generate_form_field($field_attr);
        }

        $template_page_data = str_replace('{{modal_data}}', $tmp_popup_form_field_data, $template_modal_path);
        $template_path = str_replace('{{page_modal}}', $template_page_data, $template_path);

        if (file_exists($template_modal_path) && is_readable($template_modal_path)) {
            $template_modal_path = file_get_contents($template_modal_path);
        } else {
            $template_modal_path = '';
        }
//        $template_path = str_replace('{{page_modal}}', $template_modal_path, $template_path);
    }

    return $template_path;
}
function webcheckups_manage_form_data($form_name)
{
    $url = webcheckups_api_loc . 'fielddescriptions?filter[form]=' . $form_name;

    $headers = [
        'Content-Type' => 'application/vnd.api+json',
        'Accept' => 'application/vnd.api+json',
    ];

    $response = wp_remote_get($url, ['headers' => $headers]);

    $http_status = wp_remote_retrieve_response_code($response);

    if ($http_status >= 400) {
#        error_log('Error: ' . print_r($responseData, true));
        return false;
    }

    $responseData = json_decode(wp_remote_retrieve_body($response), true);

    return $responseData['data'] ?? false;
}

function webcheckups_remove_all_api_data(): bool
{
    $token = webcheckups_get_token_from_db();
    if (!$token) {
        return true; // nothing to remove
    }
    $url = webcheckups_api_loc . 'users';

    $headers = [
        'Content-Type' => 'application/vnd.api+json',
        'Accept' => 'application/vnd.api+json',
        'Authorization' => 'Bearer ' . $token,
    ];
    $response = wp_remote_get($url, ['headers' => $headers]);

    $http_status = wp_remote_retrieve_response_code($response);

    $responseData = json_decode(wp_remote_retrieve_body($response), true);

    if ($http_status >= 400) {
#        error_log('Error: ' . print_r($responseData, true));
        return false;
    }

    $user_id = $responseData['data']['id'] ?? 0;
    if (!empty($user_id)) {
        $url = webcheckups_api_loc . 'users/' . $user_id;
        $res = wp_remote_request($url, ['method' => 'DELETE', 'headers' => $headers]);
        $http_status = wp_remote_retrieve_response_code($res);

        if ($http_status != 204) {
            return false;
        }
        return true;
    }

    return false;
}

function webcheckups_generate_form_field($field_data, $with_switch = null): string
{
    $form_name = $field_data['form'];

    $tmp_switch_data = [];

    $tmp_field_data = [
        'form_name'          => $field_data['form'],
        'field_name'         => $field_data['field'],
        'disableFor'         => $field_data['disableFor'],
        'field_type'         => $field_data['type'],
        'field_advanced'     => $field_data['advanced'],
        'field_required'     => $field_data['required'] == 1,
        'values_description' => $field_data['valuesDescription'],
        'detail'             => !empty($field_data['label']) && $field_data['required'] == 1 ?
            sanitize_text_field($field_data['label']) . ' is required' : null,
    ];

    $field_path = file_get_contents(webcheckups_lib_dir_path . 'template/field/field_' .
        $field_data['type'] . '.html');

    if ($with_switch) {
        $file_path = webcheckups_lib_dir_path . 'template/field/field_' . strtolower($field_data['type']) .
            '_with_switch.html';
        if (file_exists($file_path)) {
            $field_path                       = file_get_contents($file_path);
            $field_data['description_switch'] = 'Show all';
            $tmp_switch_data                  = [
                'form_name'          => $field_data['form'],
                'detail'             => null,
                'field_name'         => 'switch_' . $field_data['field'],
                'disableFor'         => $field_data['disableFor'],
                'field_type'         => 'switch',
                'field_advanced'     => $field_data['advanced'],
                'field_required'     => false,
                'values_description' => null,
            ];
        }
    }

    $form_key = array_search($form_name, array_column($GLOBALS['webcheckups_form_fields'], 'form_name'));
    if ($form_key !== false) {
        if ($tmp_field_data) {
            $GLOBALS['webcheckups_form_fields'][ $form_key ]['fields'][] = $tmp_field_data;
        }
        if (!empty($tmp_switch_data)) {
            $GLOBALS['webcheckups_form_fields'][ $form_key ]['fields'][] = $tmp_switch_data;
        }
    } else {
        $GLOBALS['webcheckups_form_fields'][] = [
            'form_name' => $form_name,
            'fields'    => array_filter([ $tmp_field_data, empty($tmp_switch_data) ? $tmp_switch_data : null ]),
        ];
    }

    $field_content = str_replace('{{tmp_data}}', json_encode($GLOBALS['webcheckups_form_fields']), $field_path);

    $field_content = str_replace('{{column_size}}', $field_data['width'], $field_content);

    $field_content = str_replace('{{field_id}}', $field_data['field'], $field_content);
    $field_content = str_replace('{{field_required}}', ($field_data['required'] == 1) ?
        file_get_contents(
            webcheckups_lib_dir_path . 'template/field/field_required.html'
        ) : '', $field_content);

    $field_content = str_replace('{{select_type}}', $field_data['selectType'] ?? '', $field_content);

    $field_content = str_replace('{{rounded_corner}}', $field_data['valuesDescription'] === null ?
        'rounded' : '', $field_content);
    $field_content = str_replace('{{show_tooltip}}', $field_data['valuesDescription'] === null ?
        'd-none' : '', $field_content);

    $field_content = str_replace(
        '{{field_description}}',
        $field_data['description'] ?? '',
        $field_content
    );
    $field_content = str_replace(
        '{{field_description_switch}}',
        $field_data['description_switch'] ?? '',
        $field_content
    );

    $field_content = str_replace(
        '{{subscription_info}}',
        str_contains($field_data['field'], 'subproductId') ?
            file_get_contents(webcheckups_lib_dir_path . 'template/field/field_subscription_info.html') : '',
        $field_content
    );

    return str_replace('{{field_label}}', $field_data['label'] ?? '', $field_content);
}

// NEUE Funktion zum Aktualisieren der Felderdefinitionen

function webcheckups_update_field_definitions(): bool
{
    $url = webcheckups_api_loc . 'fielddescriptions';

    $headers = [
        'Content-Type' => 'application/vnd.api+json',
        'Accept' => 'application/vnd.api+json',
    ];

    $response = wp_remote_get($url, ['headers' => $headers, 'timeout' => 30]);

    if (is_wp_error($response)) {
#        error_log('Website Checkups: Failed to fetch field definitions - ' . $response->get_error_message());
        return false;
    }

    $http_status = wp_remote_retrieve_response_code($response);

    if ($http_status !== 200) {
#        error_log('Website Checkups: API returned status ' . $http_status);
        return false;
    }

    $responseData = json_decode(wp_remote_retrieve_body($response), true);

    if (!isset($responseData['data'])) {
#        error_log('Website Checkups: Invalid API response format');
        return false;
    }

    // In wp_options speichern
    $saved = update_option(webcheckups_plugin_prefix . '_field_definitions', [
        'version' => get_plugin_data(webcheckups_root_dir_path . 'website-checkups.php')['Version'],
        'updated' => current_time('timestamp'),
        'data' => $responseData['data']
    ]);

    if ($saved) {
#        error_log('Website Checkups: Field definitions updated successfully');
        return true;
    }

    return false;
}

function webcheckups_get_field_definitions()
{
    $definitions = get_option(webcheckups_plugin_prefix . '_field_definitions');

    if (empty($definitions) || !isset($definitions['data'])) {
        // Fallback: Versuche zu aktualisieren
        webcheckups_update_field_definitions();
        $definitions = get_option(webcheckups_plugin_prefix . '_field_definitions');
    }

    return $definitions['data'] ?? [];
}
